# -*- coding: utf-8 -*-
"""
VertexDistributorUI (PRO) — UI-Frontend, gesamte Logik aus _impl.py
"""

import os, json
import maya.cmds as cmds

# Logik/Optionen aus _impl
from ._impl import run as impl_run, OPTIONS as IMPL_OPTIONS

# Ressourcen (Logo nur anzeigen, wenn vorhanden)
from . import resources as R

# Qt/Shiboken
try:
    from PySide2 import QtWidgets, QtCore, QtGui
    from shiboken2 import wrapInstance
except Exception:
    from PySide6 import QtWidgets, QtCore, QtGui  # type: ignore
    try:
        from shiboken6 import wrapInstance  # type: ignore
    except Exception:
        from shiboken import wrapInstance  # type: ignore

# OpenMayaUI
try:
    from maya import OpenMayaUI as omui  # type: ignore
except Exception:
    omui = None

CONTROL_NAME   = "VertexDistributorPRO"
WORKSPACE_NAME = CONTROL_NAME + "WorkspaceControl"

# Impl-Hooks (Actions/Setter)
_IMPL  = impl_run()                    # -> {"actions": {...}, "set_option": fn}
A      = _IMPL["actions"]
SETOPT = _IMPL["set_option"]


# ---------------- Theme (Zen-Base) ----------------
def _apply_zen_qss(widget):
    """
    Zen-Theme direkt aus pjots_zen_tools/core/ui/styles/zen_base.qss anwenden.
    Kein ProTools-Theme import.
    """
    try:
        import pjots_zen_tools as _pkg
        pkg_root = os.path.dirname(_pkg.__file__)
        qss_path = os.path.join(pkg_root, "core", "ui", "styles", "zen_base.qss")
        if os.path.exists(qss_path):
            with open(qss_path, "r", encoding="utf-8") as f:
                raw = f.read()
            widget.setStyleSheet(raw)
            return True
    except Exception:
        pass
    return False


# ---------------- Presets ----------------
def _preset_path():
    base = cmds.internalVar(usd=True)
    pdir = os.path.join(base, "pjots_zen_tools", "vdpro_presets")
    try:
        os.makedirs(pdir, exist_ok=True)
    except Exception:
        pass
    return os.path.join(pdir, "presets.json")

def _load_presets():
    p = _preset_path()
    if os.path.exists(p):
        try:
            with open(p, "r", encoding="utf-8") as f:
                data = json.load(f)
            return data if isinstance(data, list) else []
        except Exception:
            return []
    return []

def _save_presets(arr):
    try:
        with open(_preset_path(), "w", encoding="utf-8") as f:
            json.dump(arr, f, ensure_ascii=False, indent=2)
        return True
    except Exception:
        return False


# ---------------- kleine Helfer ----------------

# --- Zen-Theme Loader (ohne pro_tools) ---
def _pkg_root():
    import pjots_zen_tools as _pkg
    return os.path.dirname(_pkg.__file__)

def _zen_qss_path():
    return os.path.join(_pkg_root(), "core", "ui", "styles", "zen_base.qss")

_QSS_CACHE = None
def _load_zen_qss():
    global _QSS_CACHE
    if _QSS_CACHE is not None:
        return _QSS_CACHE
    p = _zen_qss_path()
    try:
        with open(p, "r", encoding="utf-8") as f:
            _QSS_CACHE = f.read()
    except Exception:
        _QSS_CACHE = ""
        try:
            import maya.cmds as cmds
            cmds.warning("[VDPRO] zen_base.qss not found: %s" % p)
        except Exception:
            pass
    return _QSS_CACHE



class _Row(QtWidgets.QWidget):
    def __init__(self, label, widget, parent=None):
        super().__init__(parent)
        h = QtWidgets.QHBoxLayout(self)
        h.setContentsMargins(0,0,0,0); h.setSpacing(6)
        lab = QtWidgets.QLabel(label); lab.setProperty("role", "muted")
        h.addWidget(lab, 1); h.addWidget(widget, 1)

class FloatSlider(QtWidgets.QWidget):
    valueChanged = QtCore.Signal(float)
    def __init__(self, vmin, vmax, step=0.01, decimals=3, tick_interval=None, show_value=True, parent=None):
        super().__init__(parent)
        self._vmin, self._vmax = float(vmin), float(vmax)
        self._dec = int(decimals)
        self._factor = max(1, int(round(10 ** self._dec)))
        self._step_i = max(1, int(round(step * self._factor)))
        lay = QtWidgets.QHBoxLayout(self); lay.setContentsMargins(0,0,0,0); lay.setSpacing(6)
        self._sld = QtWidgets.QSlider(QtCore.Qt.Horizontal)
        self._sld.setRange(0, int(round((self._vmax - self._vmin) * self._factor)))
        self._sld.setSingleStep(self._step_i)
        if tick_interval:
            ti = max(1, int(round(float(tick_interval) * self._factor)))
            self._sld.setTickPosition(QtWidgets.QSlider.TicksBelow)
            self._sld.setTickInterval(ti)
        lay.addWidget(self._sld, 1)
        self._lbl = QtWidgets.QLabel("") if show_value else None
        if self._lbl:
            self._lbl.setProperty("role", "muted")
            self._lbl.setAlignment(QtCore.Qt.AlignRight | QtCore.Qt.AlignVCenter)
            self._lbl.setMinimumWidth(60)
            lay.addWidget(self._lbl, 0)
        self._sld.valueChanged.connect(self._emit)

    def _fmt(self, v): return f"{v:.{self._dec}f}"
    def _emit(self, iv):
        v = self._vmin + (float(iv) / float(self._factor))
        if self._lbl: self._lbl.setText(self._fmt(v))
        self.valueChanged.emit(v)
    def setValue(self, v):
        iv = int(round((float(v) - self._vmin) * self._factor))
        self._sld.blockSignals(True); self._sld.setValue(iv); self._sld.blockSignals(False); self._emit(iv)
    def value(self):
        return self._vmin + (float(self._sld.value()) / float(self._factor))

class _Section(QtWidgets.QFrame):
    def __init__(self, title, parent=None):
        super().__init__(parent)
        self.setProperty("role", "segment")
        v = QtWidgets.QVBoxLayout(self); v.setContentsMargins(10,10,10,10); v.setSpacing(8)
        t = QtWidgets.QLabel(title); f=t.font(); f.setBold(True); t.setFont(f); v.addWidget(t)
        self.body = QtWidgets.QVBoxLayout(); self.body.setContentsMargins(0,0,0,0); self.body.setSpacing(6)
        v.addLayout(self.body)

def _make_btn(label, cb, variant=None, destructive=False, tip=""):
    b = QtWidgets.QPushButton(label)
    if tip: b.setToolTip(tip)
    if variant: b.setProperty("variant", variant)     # nutzt Zen-QSS-Varianten
    if destructive: b.setProperty("destructive", "true")
    b.setMinimumHeight(34)
    b.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)
    b.clicked.connect(cb)
    return b

def _invoke_repeatable(label: str, action_key: str):
    impl = impl_run()                 # frisches Dict
    impl["actions"][action_key](None)
    py = (
        'python("from pjots_zen_tools.tools.vertex_distributor._impl import run; '
        f'run()[\\"actions\\"][\\"{action_key}\\"](None)")'
    )
    try:
        cmds.repeatLast(ac=py, acl=f"Vertex Distributor: {label}")
    except Exception:
        pass


# ---------------- Hauptpanel ----------------
class VertexDistributorUI(QtWidgets.QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setObjectName("ZenRoot")
        self.setWindowTitle("Vertex Distributor PRO")
        self.setMinimumSize(420, 540)
        self._controls = {}
        self._build_ui()

    def _build_ui(self):
        v = QtWidgets.QVBoxLayout(self)
        v.setContentsMargins(10,10,10,10); v.setSpacing(10)

        # Logo nur anzeigen, wenn vorhanden
        logo = R.full_icon_path(R.VD_LOGO)
        if logo and os.path.exists(logo):
            lbl = QtWidgets.QLabel()
            pm = QtGui.QPixmap(logo)
            if not pm.isNull():
                pm = pm.scaledToWidth(240, QtCore.Qt.SmoothTransformation)
                lbl.setAlignment(QtCore.Qt.AlignHCenter | QtCore.Qt.AlignVCenter)
                lbl.setPixmap(pm)
                v.addWidget(lbl)

        # Options
        sec_opt = _Section("Options", self); v.addWidget(sec_opt)

        s_smooth = FloatSlider(0.00, 0.30, step=0.005, decimals=3, tick_interval=0.05, show_value=True)
        s_smooth.setValue(float(IMPL_OPTIONS.get("smooth_window_frac", 0.08)))
        s_smooth.valueChanged.connect(lambda val: SETOPT("smooth_window_frac", float(val)))
        sec_opt.body.addWidget(_Row("Smoothing", s_smooth))
        self._controls["smooth_window_frac"] = s_smooth

        s_corner = FloatSlider(0.00, 1.00, step=0.05, decimals=2, tick_interval=0.25, show_value=True)
        s_corner.setValue(float(IMPL_OPTIONS.get("corner_boost", 0.00)))
        s_corner.valueChanged.connect(lambda val: SETOPT("corner_boost", float(val)))
        sec_opt.body.addWidget(_Row("Keep Form", s_corner))
        self._controls["corner_boost"] = s_corner

        cb_even = QtWidgets.QCheckBox("Even Distribution")
        cb_even.setChecked(bool(IMPL_OPTIONS.get("even_distribution", True)))
        cb_even.stateChanged.connect(lambda _=None, w=cb_even: SETOPT("even_distribution", bool(w.isChecked())))
        sec_opt.body.addWidget(cb_even)
        self._controls["even_distribution"] = cb_even

        # Presets
        sec_p = _Section("Presets", self); v.addWidget(sec_p)
        rowp = QtWidgets.QHBoxLayout(); rowp.setSpacing(6)
        self.combo_presets = QtWidgets.QComboBox(); self.combo_presets.addItem("Load preset…", None)
        for pr in _load_presets():
            self.combo_presets.addItem(pr.get("name","preset"), pr.get("values", {}))
        btn_save = QtWidgets.QPushButton("Save…")
        btn_del  = QtWidgets.QPushButton("Delete")
        rowp.addWidget(self.combo_presets, 2); rowp.addWidget(btn_save, 0); rowp.addWidget(btn_del, 0)
        prw = QtWidgets.QWidget(); prw.setLayout(rowp)
        sec_p.body.addWidget(prw)

        def _snapshot_vals():
            return {
                "smooth_window_frac": self._controls["smooth_window_frac"].value(),
                "corner_boost":       self._controls["corner_boost"].value(),
                "even_distribution":  bool(self._controls["even_distribution"].isChecked()),
            }

        def _apply_vals(vals):
            if "smooth_window_frac" in vals:
                self._controls["smooth_window_frac"].setValue(float(vals["smooth_window_frac"]))
                SETOPT("smooth_window_frac", float(vals["smooth_window_frac"]))
            if "corner_boost" in vals:
                self._controls["corner_boost"].setValue(float(vals["corner_boost"]))
                SETOPT("corner_boost", float(vals["corner_boost"]))
            if "even_distribution" in vals:
                self._controls["even_distribution"].setChecked(bool(vals["even_distribution"]))
                SETOPT("even_distribution", bool(vals["even_distribution"]))

        self.combo_presets.currentIndexChanged.connect(
            lambda idx: (_apply_vals(dict(self.combo_presets.itemData(idx))) if self.combo_presets.itemData(idx) else None)
        )

        def _save_preset():
            name, ok = QtWidgets.QInputDialog.getText(self, "Save Preset", "Name:")
            if not ok or not name.strip(): return
            entry = {"name": name.strip(), "values": _snapshot_vals()}
            allp = _load_presets()
            lowered = name.strip().lower()
            replaced = False
            for i, pr in enumerate(allp):
                if pr.get("name","").strip().lower() == lowered:
                    allp[i] = entry; replaced = True; break
            if not replaced:
                allp.append(entry)
            if _save_presets(allp):
                self.combo_presets.blockSignals(True)
                self.combo_presets.clear(); self.combo_presets.addItem("Load preset…", None)
                sel = 0
                for i, pr in enumerate(allp, start=1):
                    self.combo_presets.addItem(pr.get("name","preset"), pr.get("values", {}))
                    if pr.get("name","").strip().lower() == lowered:
                        sel = i
                self.combo_presets.blockSignals(False)
                self.combo_presets.setCurrentIndex(sel)

        def _delete_preset():
            idx = self.combo_presets.currentIndex()
            data = self.combo_presets.itemData(idx)
            if not data: return
            name = self.combo_presets.currentText()
            allp = [p for p in _load_presets() if p.get("name") != name]
            if _save_presets(allp):
                self.combo_presets.blockSignals(True)
                self.combo_presets.clear(); self.combo_presets.addItem("Load preset…", None)
                for pr in allp:
                    self.combo_presets.addItem(pr.get("name","preset"), pr.get("values", {}))
                self.combo_presets.blockSignals(False)
                self.combo_presets.setCurrentIndex(0)

        btn_save.clicked.connect(_save_preset)
        btn_del.clicked.connect(_delete_preset)

        # Actions (2×3 Grid)
        sec_act = _Section("Actions", self); v.addWidget(sec_act)
        grid = QtWidgets.QGridLayout(); grid.setContentsMargins(0,0,0,0); grid.setHorizontalSpacing(8); grid.setVerticalSpacing(8)
        grid.setColumnStretch(0, 1); grid.setColumnStretch(1, 1)

        b1 = _make_btn("Distribute", lambda: _invoke_repeatable("Distribute", "preview_curve"), variant="primary", tip="Select Edge Chain -> Distributes Verts equally ")
        b2 = _make_btn("ReDistribute", lambda: _invoke_repeatable("ReDistribute", "distribute"), variant="success",tip="After curve change - redistributes equally" )
        b3 = _make_btn("Simplify Crv.",   lambda: _invoke_repeatable("Simplify Crv.",   "simplify_curve"),  tip="Rebuild -1 span")
        b4 = _make_btn("Complexify Crv.", lambda: _invoke_repeatable("Complexify Crv.", "complexify_curve"), tip="Rebuild +1 span")
        b5 = _make_btn("Reuse Curve (ALPHA)", lambda: _invoke_repeatable("Reuse Curve (ALPHA)", "reuse_curve"), tip="ReUses Curve on diff.Edge ALPHA")
        b6 = _make_btn("Del. Curves",     lambda: _invoke_repeatable("Del. Curves", "reset_all"), variant="warning", destructive=True, tip="Resets the HelperCurve")

        grid.addWidget(b1, 0, 0); grid.addWidget(b2, 0, 1)
        grid.addWidget(b3, 1, 0); grid.addWidget(b4, 1, 1)
        grid.addWidget(b5, 2, 0); grid.addWidget(b6, 2, 1)

        wrap = QtWidgets.QWidget(); wrap.setLayout(grid)
        sec_act.body.addWidget(wrap)

        # Footer: Help-Link
        foot = QtWidgets.QHBoxLayout()
        btn_help = QtWidgets.QPushButton("Help / Tutorial")
        btn_help.clicked.connect(lambda: QtGui.QDesktopServices.openUrl(QtCore.QUrl(R.YOUTUBE_VD)))
        foot.addStretch(1); foot.addWidget(btn_help)
        v.addLayout(foot)
        v.addStretch(1)


# ---------------- Dockbarer Entry ----------------
def show():
    import time

    # Vorhandene Instanz schließen
    if cmds.workspaceControl(WORKSPACE_NAME, q=True, exists=True):
        try:
            cmds.deleteUI(WORKSPACE_NAME)
        except Exception:
            pass

    # Neues WorkspaceControl anlegen
    cmds.workspaceControl(
        WORKSPACE_NAME,
        label="Vertex Distributor PRO",
        retain=False,
        floating=True,
        widthProperty=True,
        initialWidth=460,
        heightProperty=True,
        initialHeight=600
    )

    # Robust auffinden
    ptr = None
    for _ in range(30):
        try:
            ptr = omui.MQtUtil.findControl(WORKSPACE_NAME) if omui else None
            if not ptr and omui:
                ptr = omui.MQtUtil.findWindow(WORKSPACE_NAME)
        except Exception:
            ptr = None
        if ptr:
            break
        QtWidgets.QApplication.processEvents()
        time.sleep(0.01)

    if not ptr:
        cmds.warning("VDPRO: WorkspaceControl not found.")
        return

    host = wrapInstance(int(ptr), QtWidgets.QWidget)

    # Layout vorbereiten
    if host.layout() is None:
        host.setLayout(QtWidgets.QVBoxLayout())
    host.layout().setContentsMargins(0, 0, 0, 0)
    while host.layout().count():
        it = host.layout().takeAt(0)
        if it.widget():
            it.widget().setParent(None)

    # Root-Container (ZenRoot) einsetzen
    root = QtWidgets.QWidget(host)
    root.setObjectName("ZenRoot")
    root.setLayout(QtWidgets.QVBoxLayout())
    root.layout().setContentsMargins(10, 10, 10, 10)

    # Mindestgröße setzen
    MIN_W, MIN_H = 460, 600
    root.setMinimumSize(MIN_W, MIN_H)
    host.setMinimumSize(MIN_W, MIN_H)  # optional, unterstützt nicht jede Maya-Version

    # QSS anwenden
    qss = _load_zen_qss()
    if qss:
        root.setStyleSheet(qss)

    # Panel hinzufügen
    panel = VertexDistributorUI(parent=root)
    root.layout().addWidget(panel)
    host.layout().addWidget(root)

    # Sichtbar machen
    try:
        cmds.workspaceControl(WORKSPACE_NAME, e=True, visible=True)
        # optional: Mindestmaße auch hier setzen
        cmds.workspaceControl(WORKSPACE_NAME, e=True, minimumWidth=MIN_W, minimumHeight=MIN_H)
    except Exception:
        pass
