# src/pjots_zen_tools/core/updater.py
from __future__ import annotations

import hashlib
import json
import os
import shutil
import tempfile
import time
import zipfile
from typing import Any, Dict, Tuple
from urllib.error import HTTPError, URLError
from urllib.parse import urljoin, urlparse
from urllib.request import urlopen

from .settings import get_update_base_url

# --------------------------------------------------------------------------------------
# Paths
# --------------------------------------------------------------------------------------


def _pkg_root() -> str:
    """Return package root directory (…/pjots_zen_tools)."""
    return os.path.dirname(os.path.dirname(__file__))


def _tools_root() -> str:
    return os.path.join(_pkg_root(), "tools")


# --------------------------------------------------------------------------------------
# Helpers
# --------------------------------------------------------------------------------------


def _calc_sha256(path: str) -> str:
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(65536), b""):
            h.update(chunk)
    return h.hexdigest()


def _semver_tuple(v: str) -> Tuple[int, int, int]:
    """Simple semver-ish: "1.2.3" → (1, 2, 3); non-numeric sanitized."""
    parts = (v or "0").split(".")
    out = []
    for p in parts[:3]:
        try:
            out.append(int("".join(ch for ch in p if ch.isdigit())))
        except Exception:
            out.append(0)
    while len(out) < 3:
        out.append(0)
    return tuple(out)  # type: ignore[return-value]


def _ensure_abs_url(zip_url: str, base_url: str | None) -> str:
    if not zip_url:
        return zip_url
    if bool(urlparse(zip_url).scheme):  # already absolute
        return zip_url
    if base_url:
        return urljoin(base_url.rstrip("/") + "/", zip_url.lstrip("/"))
    return zip_url


# --------------------------------------------------------------------------------------
# Manifest
# --------------------------------------------------------------------------------------


def fetch_manifest(manifest_url: str | None = None) -> tuple[dict, str]:
    """Load manifest and return (manifest_dict, base_url).

    Supports two shapes:
      - {"modules": [{"module_id", "version", "zip_url", "sha256"}]}
      - {"tools":   [{"id",         "version", "zip_url", "sha256"}]}
    """
    base = (manifest_url or get_update_base_url()).rstrip("/") + "/"
    url = base + "manifest.json"
    with urlopen(url, timeout=10) as r:
        data = r.read().decode("utf-8")
    return json.loads(data) or {}, base


def check_updates(local_registry: dict, manifest_url: str | None = None):
    """Vergleicht Registry mit Server-Manifest.

    Returns list[dict] with keys:
      module_id, title, local_version, remote_version, zip_url, checksum
    """
    try:
        manifest, base = fetch_manifest(manifest_url)
    except (URLError, HTTPError, TimeoutError) as e:  # type: ignore[name-defined]
        return {"error": f"Manifest fetch failed: {e}"}

    # Normalize manifest items (support tools/modules)
    items = []
    if isinstance(manifest, dict):
        if isinstance(manifest.get("modules"), list):
            items = manifest["modules"]
        elif isinstance(manifest.get("tools"), list):
            # map "tools" structure to our canonical format
            for t in manifest["tools"]:
                items.append(
                    {
                        "module_id": t.get("id"),
                        "version": t.get("version"),
                        "zip_url": t.get("zip_url"),
                        "sha256": t.get("sha256"),
                        "title": t.get("title", t.get("id")),
                    }
                )

    srv: Dict[str, Dict[str, Any]] = {m.get("module_id"): m for m in items if m.get("module_id")}

    out = []
    for mid, meta in (local_registry or {}).items():
        local_v = meta.get("version") or "0.0.0"
        remote = srv.get(mid)
        if not remote:
            continue
        remote_v = remote.get("version") or "0.0.0"
        if _semver_tuple(remote_v) > _semver_tuple(local_v):
            zip_url = remote.get("zip_url")
            zip_url = _ensure_abs_url(zip_url, base)
            out.append(
                {
                    "module_id": mid,
                    "title": meta.get("title", mid) or remote.get("title", mid),
                    "local_version": local_v,
                    "remote_version": remote_v,
                    "zip_url": zip_url,
                    # normalize checksum key name
                    "checksum": (remote.get("sha256") or remote.get("checksum") or ""),
                }
            )
    return out


# --------------------------------------------------------------------------------------
# Install / Uninstall
# --------------------------------------------------------------------------------------


def install_zip_to_tool(zip_path: str, module_id: str) -> str:
    """Entpackt ZIP in tools/<module_id>/ (overwrite) und gibt Zielpfad zurück."""
    tools_root = _tools_root()
    target = os.path.join(tools_root, module_id)
    os.makedirs(target, exist_ok=True)
    with zipfile.ZipFile(zip_path, "r") as z:
        z.extractall(target)
    return target


def download_and_install(update_item: dict, manifest_url: str | None = None) -> str:
    """Lädt ZIP, prüft optional sha256, installiert. Returns message str.

    Compatible with callers that pass either "sha256" or "checksum" in update_item.
    Also accepts an optional manifest_url for resolving relative zip URLs.
    """
    url = update_item.get("zip_url")
    mid = update_item.get("module_id")
    want_sha = (update_item.get("sha256") or update_item.get("checksum") or "").lower().strip()

    if not url or not mid:
        return "Invalid update item."

    # ensure absolute url if manifest_url (base) was given
    url = _ensure_abs_url(url, manifest_url)

    with tempfile.TemporaryDirectory() as td:
        tmp_zip = os.path.join(td, f"{mid}.zip")
        with urlopen(url, timeout=30) as r, open(tmp_zip, "wb") as f:
            f.write(r.read())

        if want_sha:
            got_sha = _calc_sha256(tmp_zip)
            if got_sha.lower() != want_sha:
                return (
                    f"Checksum mismatch for {mid}! expected {want_sha}, got {got_sha}"
                )

        target = install_zip_to_tool(tmp_zip, mid)

    return (
        f"Updated {mid} → installed to: {target}\n"
        f"Please restart Maya (or reload module) to apply."
    )


def _rmtree_force(path: str) -> None:
    """Robustes Entfernen eines Verzeichnisses (Windows-Locks berücksichtigen)."""
    def _onerror(func, p, exc_info):  # noqa: ANN001
        try:
            os.chmod(p, 0o700)
            func(p)
        except Exception:
            pass

    for i in range(2):  # kleiner Retry hilft bei kurzzeitig gelockten Dateien
        try:
            shutil.rmtree(path, onerror=_onerror)
            return
        except Exception:
            time.sleep(0.1)
    # letzter Versuch ohne onerror (wir wollen nicht hart fehlschlagen)
    try:
        shutil.rmtree(path)
    except Exception:
        pass


def uninstall(module_id: str | None = None, path: str | None = None) -> str:
    """Unified uninstall API used by Hub.

    Supports both signatures used in the wild:
      - uninstall(module_id=...)  (preferred)
      - uninstall("module_id")   (legacy positional)

    If a path is passed, that directory will be removed directly.
    Otherwise we resolve tools/<module_id>.
    Returns a short status message.
    """
    # legacy positional arg shim: if module_id is None but path holds a string without os.sep
    if module_id is None and path and os.sep not in path:
        module_id = path
        path = None

    if not module_id and not path:
        return "Uninstall: no module_id or path provided."

    if path is None and module_id:
        path = os.path.join(_tools_root(), module_id)

    if not path or not os.path.exists(path):
        return f"Module not found: {module_id or path}"

    _rmtree_force(path)
    return f"Uninstalled {module_id or path}."
