# -*- coding: utf-8 -*-
"""
Pjots Zen Tools – Drag&Drop Installer für Maya
Datei ins Maya-Viewport ziehen ODER manuell ausführen.
DIESER INSTALLER:
- Kopiert Paket + Icons
- Erstellt/aktualisiert Live-Shelf (mit ann="zen:*")
- Schreibt eine MINIMALE shelf_PjotsZenTools.mel, die beim Start
  Python aufruft und das Shelf dynamisch aus der Registry aufbaut.
"""

from __future__ import print_function, unicode_literals
import os
import sys
import shutil
import datetime

try:
    import maya.cmds as cmds
    import maya.mel as mel
except Exception:
    cmds = None
    mel = None

# ------------------------------
# Helpers
# ------------------------------
def _safe_makedirs(path):
    try:
        if not os.path.isdir(path):
            os.makedirs(path)
    except OSError:
        pass

# ------------------------------
# Subtle UI messaging (less intrusive)
# ------------------------------
QUIET_UI = False          # auf True setzen, wenn du gar keine inViewMessages willst
INVIEW_POS = "topCenter"   # "topRight", "botRight", "botLeft", "topCenter"

def _inview(msg, stay_ms=900):
    """Sehr dezente In-View-Message ohne gelbes Highlight."""
    if not cmds or QUIET_UI:
        return
    try:
        # KEIN <hl> mehr und KEIN 'amg' -> verhindert gelbe Box
        cmds.inViewMessage(
            msg=str(msg),
            pos=INVIEW_POS,
            fade=True,
            fadeStayTime=int(stay_ms)  # kurz & dezent
        )
    except Exception:
        pass

def info(msg):
    """Konsole + dezentes HUD statt gelber Box."""
    try:
        print("[PZT] " + str(msg))
        _inview(msg, stay_ms=900)
    except Exception:
        pass

def warn(msg):
    """Etwas länger sichtbar, aber weiterhin dezent."""
    try:
        print("[PZT][WARN] " + str(msg))
        _inview("⚠ " + str(msg), stay_ms=1400)
    except Exception:
        pass

def error(msg):
    """Nur bei echten Fehlern etwas prominenter, aber ohne Gelb."""
    try:
        print("[PZT][ERROR] " + str(msg))
        _inview("✖ " + str(msg), stay_ms=1800)
    except Exception:
        pass

def get_install_root_from(file_path_hint):
    try:
        if file_path_hint:
            return os.path.dirname(os.path.realpath(file_path_hint))
    except Exception:
        pass
    try:
        return os.path.dirname(os.path.realpath(__file__))
    except Exception:
        return os.getcwd()

def get_user_maya_paths():
    maya_version = None
    try:
        if cmds:
            maya_version = cmds.about(v=True)
    except Exception:
        pass
    if not maya_version:
        maya_version = os.environ.get("MAYA_VERSION", "2026")  # default 2026

    maya_app_dir = os.environ.get("MAYA_APP_DIR")
    if not maya_app_dir:
        home = os.path.expanduser("~")
        if sys.platform.startswith("win"):
            maya_app_dir = os.path.join(home, "Documents", "maya")
        elif sys.platform == "darwin":
            maya_app_dir = os.path.join(home, "Library", "Preferences", "Autodesk", "maya")
        else:
            maya_app_dir = os.path.join(home, "maya")

    user_scripts = os.path.join(maya_app_dir, "scripts")
    prefs_root   = os.path.join(maya_app_dir, str(maya_version), "prefs")
    user_icons   = os.path.join(prefs_root, "icons")
    user_shelves = os.path.join(prefs_root, "shelves")

    for p in (user_scripts, user_icons, user_shelves):
        _safe_makedirs(p)

    return dict(
        maya_version=str(maya_version),
        maya_app_dir=maya_app_dir,
        scripts=user_scripts,
        icons=user_icons,
        shelves=user_shelves,
    )

def copy_tree(src, dst):
    if not os.path.exists(src):
        return
    _safe_makedirs(dst)
    for root, dirs, files in os.walk(src):
        rel = os.path.relpath(root, src)
        target_root = os.path.join(dst, rel) if rel != "." else dst
        _safe_makedirs(target_root)
        for f in files:
            shutil.copy2(os.path.join(root, f), os.path.join(target_root, f))

def backup_file(path):
    if os.path.isfile(path):
        ts = datetime.datetime.now().strftime("%Y%m%d_%H%M%S")
        bak = path + ".bak_" + ts
        shutil.copy2(path, bak)
        return bak
    return None

def _delete_legacy_shelf_names():
    """Alte Shelf-Namen (mit Leerzeichen/Underscore) zur Sicherheit entfernen."""
    if not cmds:
        return
    for old in ("Pjots Zen Tools", "Pjots_Zen_Tools"):
        try:
            if cmds.shelfLayout(old, exists=True):
                cmds.deleteUI(old, layout=True)
        except Exception:
            pass

def _ensure_shelf_layout(top):
    if not cmds:
        return
    if not cmds.shelfLayout(SHELF_NAME, exists=True):
        cmds.shelfLayout(SHELF_NAME, parent=top)
    else:
        try:
            cmds.setParent(SHELF_NAME)
        except Exception:
            pass

def _remove_duplicate_buttons_by_labels(labels):
    if not cmds:
        return
    children = cmds.shelfLayout(SHELF_NAME, q=True, ca=True) or []
    for c in children:
        try:
            if cmds.objectTypeUI(c) == "shelfButton":
                lbl = cmds.shelfButton(c, q=True, label=True)
                if lbl in labels:
                    cmds.deleteUI(c, control=True)
        except Exception:
            pass

# -------- Registry-first command builder (Option B) --------
def build_registry_first_command(module_id: str, fallback_entry: str) -> str:
    """
    Python-Code (als String) für Shelf-Buttons:
      1) Registry → launcher() falls vorhanden
      2) Fallback → importlib auf 'fallback_entry' (e.g. 'pkg.mod:func')
    """
    mod, func = fallback_entry.split(":", 1)
    return (
        "import importlib\n"
        "import pjots_zen_tools.hub.registry as _reg\n"
        f"_mid='{module_id}'\n"
        "_m=_reg.get_tools().get(_mid)\n"
        "_l=_m.get('launcher') if _m else None\n"
        "if callable(_l):\n"
        "    _l()\n"
        "else:\n"
        f"    _m = importlib.import_module('{mod}')\n"
        f"    getattr(_m, '{func}')()\n"
    )

# ------------------------------
# Konfiguration
# ------------------------------
PACKAGE_NAME = "pjots_zen_tools"

SHELF_NAME = "PjotsZenTools"                 # konsistenter, leerzeichenloser Name
SHELF_FILE_BASENAME = "shelf_PjotsZenTools.mel"
SHELF_PROC_NAME = os.path.splitext(SHELF_FILE_BASENAME)[0]  # "shelf_PjotsZenTools"

# --- Hub (Button 1) ---
HUB_BUTTON_LABEL = "ZenTools Hub"
HUB_BUTTON_ANNOT = "Open ZenTools Hub"
HUB_BUTTON_ICON  = "pjots_zen_tools_hub.png"   # bitte in /icons/ bereitstellen
HUB_COMMAND      = build_registry_first_command(
    module_id="zen_hub",
    fallback_entry="pjots_zen_tools.hub.hub_ui:show"
)
HUB_ANN          = "zen:hub"

# --- Vertex Distributor (Button 2) ---
SHELF_BUTTON_LABEL = "Vertex Distributor"
SHELF_BUTTON_ANNOT = "Vertex Distributor - distributes verts along a curve"
SHELF_BRAND_ICON   = "pjots_zen_tools.png"     # Fallback/Brand
SHELF_TOOL_ICON    = "vertex_distributor.png"
SHELF_COMMAND      = build_registry_first_command(
    module_id="vertex_distributor",
    fallback_entry="pjots_zen_tools.tools.vertex_distributor.launch:show"
)
VD_ANN             = "zen:vertex_distributor"

# Effektive Icons (werden in run_install gesetzt)
SHELF_BUTTON_ICON_EFFECTIVE = SHELF_BRAND_ICON
HUB_BUTTON_ICON_EFFECTIVE   = HUB_BUTTON_ICON

def ensure_shelf_in_session():
    """
    Live-Shelf anlegen/aktualisieren, wenn UI bereit ist.
    Reihenfolge: 1) Hub, 2) Vertex Distributor.
    **Mit ann="zen:*"**, damit spätere Uninstalls sie zuverlässig finden.
    """
    if not cmds or cmds.about(batch=True):
        return

    try:
        top = mel.eval('global string $gShelfTopLevel; $gShelfTopLevel;') if mel else None
    except Exception:
        top = None
    if not top or not cmds.control(top, exists=True):
        return

    _delete_legacy_shelf_names()
    try:
        cmds.setParent(top)
    except Exception:
        pass

    _ensure_shelf_layout(top)

    # Duplikate (beide Labels) entfernen
    _remove_duplicate_buttons_by_labels({HUB_BUTTON_LABEL, SHELF_BUTTON_LABEL})

    # --- 1) Hub-Button ---
    try:
        cmds.shelfButton(
            parent=SHELF_NAME,
            i=HUB_BUTTON_ICON_EFFECTIVE,
            l=HUB_BUTTON_LABEL,
            ann=HUB_ANN,
            stp="python",
            c=HUB_COMMAND,
            imageOverlayLabel=HUB_BUTTON_LABEL[:4],
        )
    except Exception as e:
        info("Hub Button could not be created: {}".format(e))

    # --- 2) Vertex Distributor ---
    try:
        cmds.shelfButton(
            parent=SHELF_NAME,
            i=SHELF_BUTTON_ICON_EFFECTIVE,
            l=SHELF_BUTTON_LABEL,
            ann=VD_ANN,
            stp="python",
            c=SHELF_COMMAND,
            imageOverlayLabel=SHELF_BUTTON_LABEL[:4],
        )
    except Exception as e:
        info("VD Button could not be created: {}".format(e))

def _ascii_safe(s):
    repl = {
        u"ä": "ae", u"ö": "oe", u"ü": "ue",
        u"Ä": "Ae", u"Ö": "Oe", u"Ü": "Ue", u"ß": "ss",
        u"–": "-", u"—": "-", u"„": '"', u"“": '"',
        u"‚": "'", u"’": "'", u"«": '"', u"»": '"'
    }
    try:
        for k, v in repl.items():
            s = s.replace(k, v)
        return s.encode("ascii", "ignore").decode("ascii")
    except Exception:
        return str(s).encode("ascii", "ignore").decode("ascii")

def write_minimal_persistent_shelf(paths):
    """
    Schreibt eine MINIMALE shelf_PjotsZenTools.mel, die nur:
      - das Shelf existieren lässt
      - DANN Python aufruft: build_or_sync(name, bootstrap=['hub','vd'])
    Damit ist die Persistenz immer konsistent mit Registry/Uninstalls.
    """
    shelf_file = os.path.join(paths["shelves"], SHELF_FILE_BASENAME)
    backup_file(shelf_file)

    shelf_name = _ascii_safe(SHELF_NAME)
    proc_name  = _ascii_safe(SHELF_PROC_NAME)

    # Python-Aufruf string-escaped
    py_call = (
        'python("import pjots_zen_tools.core.shelves as _s; '
        '_s.build_or_sync(name=\\"' + shelf_name + '\\", bootstrap=[\\"hub\\", \\"vd\\"])");'
    )

    mel_lines = [
        'global proc {proc}() {{'.format(proc=proc_name),
        '    global string $gShelfTopLevel;',
        '    string $shelfName = "{name}";'.format(name=shelf_name),
        '    if (!`shelfLayout -exists $shelfName`) {',
        '        shelfLayout $shelfName;',
        '    }',
        '    setParent $shelfName;',
        '    // Dynamischer Python-Builder: legt Buttons an und synchronisiert mit Registry',
        '    {py}'.format(py=py_call),
        '}',
        ''
    ]

    with open(shelf_file, "w", encoding="ascii", errors="ignore") as f:
        f.write("\n".join(mel_lines))
    return shelf_file

def source_and_build_shelf(paths):
    """
    Lädt die eben geschriebene shelf_*.mel und ruft die Proc auf,
    damit das Shelf sofort befüllt wird (ohne Neustart).
    """
    if not mel or not cmds or cmds.about(batch=True):
        return
    mel_path = os.path.join(paths["shelves"], SHELF_FILE_BASENAME)
    mel_path_fwd = mel_path.replace("\\", "/")
    try:
        mel.eval('source "{p}"; {proc}();'.format(p=mel_path_fwd, proc=SHELF_PROC_NAME))
        info("Shelf MEL sourced and built.")
    except Exception as e:
        info("Konnte Shelf MEL nicht source/callen: {}".format(e))

def remove_legacy_files(paths):
    """
    Entfernt fehlerhafte doppelte/alte Shelf-Dateien:
      - shelf_PjotsZenTools.mel.mel
      - PjotsZenTools.mel (falls jemals erzeugt)
    """
    suspects = [
        os.path.join(paths["shelves"], "shelf_PjotsZenTools.mel.mel"),
        os.path.join(paths["shelves"], "PjotsZenTools.mel"),
    ]
    for p in suspects:
        try:
            if os.path.isfile(p):
                os.remove(p)
                info("Legacy shelf file removed: {}".format(p))
        except Exception:
            pass

def run_install(install_root):
    paths = get_user_maya_paths()

    info("Installationsordner: {}".format(install_root))
    info("Maya Scripts: {}".format(paths["scripts"]))
    info("Maya Icons:   {}".format(paths["icons"]))
    info("Maya Shelves: {}".format(paths["shelves"]))

    # 1) Package kopieren
    src_pkg_root = os.path.join(install_root, "src")
    if not os.path.isdir(src_pkg_root):
        raise RuntimeError("src/-Folder not found")
    info("Installing Scripts/ ...")
    copy_tree(src_pkg_root, paths["scripts"])

    # 2) Icons kopieren
    icons_src = os.path.join(install_root, "icons")
    if os.path.isdir(icons_src):
        info("Installing Icons/ ...")
        copy_tree(icons_src, paths["icons"])
    else:
        info("No Icons folder found.")

    # 2b) Effektive Button-Icons bestimmen (Fallbacks, falls einzelne fehlen)
    global SHELF_BUTTON_ICON_EFFECTIVE, HUB_BUTTON_ICON_EFFECTIVE
    vd_icon_abs = os.path.join(paths["icons"], SHELF_TOOL_ICON)
    hub_icon_abs = os.path.join(paths["icons"], HUB_BUTTON_ICON)

    SHELF_BUTTON_ICON_EFFECTIVE = SHELF_TOOL_ICON if os.path.exists(vd_icon_abs) else SHELF_BRAND_ICON
    HUB_BUTTON_ICON_EFFECTIVE   = HUB_BUTTON_ICON  if os.path.exists(hub_icon_abs) else SHELF_BRAND_ICON

    # 3) Legacy-Dateien bereinigen (vor MEL schreiben)
    remove_legacy_files(paths)

    # 4) Live-Shelf anlegen (optional, falls UI bereit) – mit ann="zen:*"
    ensure_shelf_in_session()

    # 5) Persistente shelf-Datei schreiben (minimal, ruft Python-Builder)
    shelf_file = write_minimal_persistent_shelf(paths)
    info("Shelf created: {}".format(shelf_file))

    # 6) Sofort bauen (source + proc call)
    source_and_build_shelf(paths)

    # 7) Fertigmeldung
    msg = "Pjots Zen Tools installed.\n\nShelf: {shelf}\nPackage: {pkg}\nIcons: {icons}".format(
        shelf=SHELF_NAME, pkg=paths["scripts"], icons=paths["icons"]
    )
    info("Installation completed. Restart Maya to persist the shelf.")
    print(msg)
    # Falls du den Dialog trotzdem willst, setze SHOW_DIALOG=True
    SHOW_DIALOG = False
    if cmds and SHOW_DIALOG:
        try:
            cmds.confirmDialog(t="Installation completed", m=msg, button=["OK"])
        except Exception:
            pass

# ------------------------------
# Entry Points
# ------------------------------
def onMayaDroppedPythonFile(*args):
    file_path = args[0] if args else None
    install_root = get_install_root_from(file_path)
    run_install(install_root)

def main():
    here = get_install_root_from(__file__)
    run_install(here)

if __name__ in ("__main__", "__mp_main__"):
    main()
