# -*- coding: utf-8 -*-
import re, math
import maya.cmds as cmds
import maya.api.OpenMaya as om
from contextlib import contextmanager

@contextmanager
def _undo_chunk(label="VDPRO"):
    try:
        cmds.undoInfo(openChunk=True, cn=label)
    except Exception:
        pass
    try:
        yield
    finally:
        try:
            cmds.undoInfo(closeChunk=True)
        except Exception:
            pass


CURVE_PREFIX = "VDPRO_Curve"
TAG_ATTR, TAG_VALUE = "vdproTag", "1"

OPTIONS = {
    "auto_apex": True,               # Mid/Apex automatisch bestimmen
    "keep_curves_on_reset": False,   # Kurven bei Reset behalten?
    "apex_end_padding_frac": 0.08,       # % der Kette an jedem Ende ignorieren
    "apex_centrality_sigma_frac": 0.22,  # Breite des Gauß-Boosts um die Kettenmitte
    "apex_mix": 0.7,                     # 0..1: Anteil Chord-Distance vs. Curvature
    "preview_degree": 3,            # NURBS-Grad (1..3 sinnvoll)
    "preview_points": 32,           # Zielanzahl EPs (wird gekappt)
    "min_preview_points": 6,        # Untergrenze EPs
    "max_preview_points": 64,       # Obergrenze EPs
    "smooth_window_frac": 0.08,     # Gleitfenster (relativ zur Kettenlänge, odd)
    "corner_boost": 0.0,            # 0..1; >0 fügt extra Samples rund um Apex ein
    "reuse_anchor": "start",   # "start" oder "end": an welches Kettenende soll der Kurven-Start andocken?
    "reuse_auto_distribute": True,
    "even_distribution": True,
}

def set_option(k, v):
    OPTIONS[k] = v

STATE = {
    "edges": [],        # flache Edge-Komponenten (["mesh.e[3]", ...]) – nur für Preview/ReUse
    "start": None,      # "mesh.vtx[i]"
    "mid":   None,      # "
    "end":   None,      # "
    "curve": None,      # Transform-Name der VDPRO-Kurve
    "last_mesh": None,  # Meshname der zuletzt verwendeten Kette
    "ordered": [],      # **GESPEICHERTE Vertex-Komponenten in Reihenfolge** (für Distribute)
}

# ---------- Utilities ----------
def _log(msg): print(f"[VDPRO] {msg}")

def _cum_lengths_from_positions(points: list[om.MVector]):
    """Kumulierte Längen entlang einer Punktliste.
    Rückgabe (cum, total): cum[i] ist Länge vom Start bis Punkt i (cum[0]==0)."""
    if not points:
        return [0.0], 0.0
    cum = [0.0]
    total = 0.0
    for i in range(1, len(points)):
        total += (points[i] - points[i-1]).length()
        cum.append(total)
    return cum, total

def _last_vdpro_curve():
    curvs = _list_vdpro_curves()
    if not curvs:
        return None
    def _k(n):
        m = re.search(r'_(\d+)$', n)
        return int(m.group(1)) if m else -1
    curvs.sort(key=_k)
    return curvs[-1]

def _rebuild_curve_relative(crv_xf, delta_spans):
    fn = _mfn_curve_from_transform(crv_xf)
    if not fn:
        _log("Ungültige Kurve.")
        return False
    spans = int(fn.numSpans)
    deg   = int(fn.degree)
    new_s = max(deg, spans + int(delta_spans))  # spans >= degree absichern
    try:
        cmds.rebuildCurve(crv_xf,
                          ch=False, rpo=True,
                          rt=0, end=1, kr=0, kcp=False, kep=True, kt=False,
                          s=new_s, d=deg, tol=0.0001, keepRange=2)
        _ensure_tag_attr(crv_xf)  # Tag sicherstellen
        STATE["curve"] = crv_xf
        return True
    except Exception as e:
        _log(f"Rebuild failed: {e}")
        return False

def simplify_curve(ctx=None, **opts):
    _apply_options(opts)
    crv = STATE.get("curve")
    if not (crv and cmds.objExists(crv) and _is_vdpro_curve(crv)):
        crv = _last_vdpro_curve()
    if not crv:
        _log("Keine VDPRO-Kurve gefunden.")
        return False
    ok = _rebuild_curve_relative(crv, -1)
    if not ok: return False
    try:
        return distribute(ctx)  # nachziehen
    except Exception:
        return True

def complexify_curve(ctx=None, **opts):
    _apply_options(opts)
    crv = STATE.get("curve")
    if not (crv and cmds.objExists(crv) and _is_vdpro_curve(crv)):
        crv = _last_vdpro_curve()
    if not crv:
        _log("Keine VDPRO-Kurve gefunden.")
        return False
    ok = _rebuild_curve_relative(crv, +1)
    if not ok: return False
    try:
        return distribute(ctx)
    except Exception:
        return True

def _apply_options(opts):
    if not opts: return
    for k, v in opts.items():
        if k in OPTIONS:  # nur bekannte Optionen übernehmen
            OPTIONS[k] = v

def _maybe_reverse_vertex_order_for_better_fit(fn: om.MFnNurbsCurve, sample_max: int = 48) -> bool:
    """
    Vergleicht die Passung 'ordered' vs. 'reversed(ordered)' gegen die ausgerichtete Kurve.
    Nutzt Arc-Length-Samples (inkl. Endpunktgewicht). Reverset STATE['ordered'] nur,
    wenn die reversed-Variante deutlich besser ist. Rückgabe: True, wenn reversed wurde.
    """
    ordered = STATE.get("ordered") or []
    n = len(ordered)
    if n < 3 or fn is None:
        return False

    L = fn.length()
    if L < 1e-8:
        return False

    # gleichmäßig verteilte Indizes (max sample_max Stück)
    if n <= sample_max:
        idxs = list(range(n))
    else:
        step = max(1, int(n / sample_max))
        idxs = list(range(0, n, step))
        if idxs[-1] != n - 1:
            idxs.append(n - 1)

    def _err(rev: bool) -> float:
        err = 0.0
        for i in idxs:
            t = i / float(n - 1)
            u = fn.findParamFromLength(L * t)
            pt = fn.getPointAtParam(u, om.MSpace.kWorld)  # MPoint
            # zu vergleichender Vertex-Index
            vi = (n - 1 - i) if rev else i
            vx, vy, vz = cmds.pointPosition(ordered[vi], w=True)
            dv = om.MVector(vx - pt.x, vy - pt.y, vz - pt.z)
            err += dv * dv  # quadratisch, stabiler
        # Endpunkte stärker gewichten
        p0 = fn.getPointAtParam(fn.knotDomain[0], om.MSpace.kWorld)
        p1 = fn.getPointAtParam(fn.knotDomain[1], om.MSpace.kWorld)
        a0x, a0y, a0z = cmds.pointPosition(ordered[0 if not rev else -1],  w=True)
        a1x, a1y, a1z = cmds.pointPosition(ordered[-1 if not rev else 0], w=True)
        err += 3.0 * (om.MVector(a0x - p0.x, a0y - p0.y, a0z - p0.z) * om.MVector(a0x - p0.x, a0y - p0.y, a0z - p0.z))
        err += 3.0 * (om.MVector(a1x - p1.x, a1y - p1.y, a1z - p1.z) * om.MVector(a1x - p1.x, a1y - p1.y, a1z - p1.z))
        return float(err)

    err_keep = _err(False)
    err_rev  = _err(True)

    if err_rev + 1e-6 < err_keep:
        STATE["ordered"] = list(reversed(ordered))
        return True
    return False



def _sel_edges(edges):
    try:
        if edges:
            cmds.select(edges, r=True)
    except Exception:
        pass

def _world_positions(mesh, indices):
    return [_vtx_world(mesh, i) for i in indices]

def _farthest_pair(pos):
    # O(n^2) genügt (Loops selten riesig)
    n = len(pos)
    best = (0.0, 0, 0)
    for i in range(n):
        pi = pos[i]
        for j in range(i+1, n):
            d = (pos[j] - pi).length()
            if d > best[0]:
                best = (d, i, j)
    return best[1], best[2]

def _flip_if_better(crv, a_new: om.MVector, b_new: om.MVector) -> bool:
    fn = _mfn_curve_from_transform(crv)
    if not fn:
        return False

    # Start/End & Richtungen
    u0, u1 = fn.knotDomain
    p0 = om.MVector(fn.getPointAtParam(u0, om.MSpace.kWorld))
    p1 = om.MVector(fn.getPointAtParam(u1, om.MSpace.kWorld))
    v_curve = p1 - p0
    v_chain = b_new - a_new

    if v_curve.length() < 1e-8 or v_chain.length() < 1e-8:
        return False

    vcN = v_curve / v_curve.length()
    vnN = v_chain / v_chain.length()

    # (1) Primärer Test: Richtungs-Dot
    flip = (vcN * vnN) < -1e-6  # klar entgegen gesetzt?

    # (2) Fallback/Absicherung: 3 Samples entlang der Kurve vs. Segment
    if not flip:
        try:
            L = fn.length()
            if L > 1e-6:
                fracs = (0.25, 0.5, 0.75)
                err_ok = 0.0
                err_rev = 0.0
                for f in fracs:
                    u = fn.findParamFromLength(L * f)
                    P = om.MVector(fn.getPointAtParam(u, om.MSpace.kWorld))
                    E_ok  = a_new + v_chain * f
                    E_rev = b_new - v_chain * f
                    err_ok  += (P - E_ok).length()
                    err_rev += (P - E_rev).length()
                if err_rev + 1e-4 < err_ok:
                    flip = True
        except Exception:
            # Wenn Sampling schiefgeht, reicht der Dot-Test
            pass

    if flip:
        # Kurve spiegeln
        cmds.reverseCurve(crv, ch=False, rpo=True)

        # STATE['ordered'] synchron spiegeln (für Distribute)
        if STATE.get("ordered"):
            STATE["ordered"] = list(reversed(STATE["ordered"]))

        # Pivot neu auf Kurvenstart setzen und Start exakt auf a_new snappen
        fn = _mfn_curve_from_transform(crv)
        p0n = om.MVector(fn.getPointAtParam(fn.knotDomain[0], om.MSpace.kWorld))
        cmds.xform(crv, ws=True, piv=(p0n.x, p0n.y, p0n.z))
        delta = a_new - p0n
        cmds.xform(crv, ws=True, t=(delta.x, delta.y, delta.z), r=True)
        return True

    return False



def _linearize_ring(indices, i_start, i_end):
    """Ring in offene Kette umwandeln: von i_start bis i_end vorwärts (mit Wrap)."""
    n = len(indices)
    if i_start <= i_end:
        return indices[i_start:i_end+1]
    # wrap
    return indices[i_start:] + indices[:i_end+1]

def _maybe_break_closed_chain(mesh, ordered_indices, is_closed):
    """
    - Offene Kette: gibt (start_idx, end_idx, ordered_indices) zurück.
    - Geschlossene Kette: bricht am weitest entfernten Paar auf.
    """
    if not ordered_indices:
        return None, None, []
    if not is_closed:
        return ordered_indices[0], ordered_indices[-1], ordered_indices

    # Loop → am weitesten Paar aufbrechen (stabil gegen schräge Edge-Reihenfolgen)
    pos = _world_positions(mesh, ordered_indices)
    i, j = _farthest_pair(pos)
    linear = _linearize_ring(ordered_indices, i, j)
    return linear[0], linear[-1], linear


def _remember_edges_from_selection():
    """Aktuelle Edge-Selektion in STATE.edges übernehmen (falls vorhanden)."""
    sel = cmds.ls(sl=True, fl=True) or []
    edges = cmds.filterExpand(sel, sm=32) or []
    if edges:
        STATE["edges"] = edges
        STATE["last_mesh"] = edges[0].split(".e[",1)[0]

def _ensure_tag_attr(node):
    if not cmds.objExists(node): return
    if not cmds.attributeQuery(TAG_ATTR, n=node, exists=True):
        cmds.addAttr(node, ln=TAG_ATTR, dt="string")
    cmds.setAttr(f"{node}.{TAG_ATTR}", TAG_VALUE, type="string")

def _is_vdpro_curve(xform):
    if not cmds.objExists(xform) or cmds.nodeType(xform) != "transform":
        return False
    if cmds.attributeQuery(TAG_ATTR, n=xform, exists=True):
        try:
            return cmds.getAttr(f"{xform}.{TAG_ATTR}") == TAG_VALUE
        except Exception:
            pass
    return False

def _list_vdpro_curves():
    cand = cmds.ls(f"{CURVE_PREFIX}*", type="transform") or []
    return [c for c in cand if _is_vdpro_curve(c)]

def _unique_curve_name():
    i = 1
    while True:
        n = f"{CURVE_PREFIX}_{i:03d}"
        if not cmds.objExists(n): return n
        i += 1

def _edge_endpoints(edge):
    try:
        m, rng = edge.split(".e[", 1)
        a, b = rng[:-1].split(":", 1) if ":" in rng else (rng[:-1], rng[:-1])
        if a == b:
            v = cmds.polyInfo(edge, ev=True) or []
            if not v: return (None, None)
            nums = [int(t) for t in re.findall(r"\d+", v[0])]
            return (nums[-2], nums[-1])
        else:
            eids = cmds.filterExpand([edge], sm=32) or []
            if not eids: return (None, None)
            v = cmds.polyInfo(eids[0], ev=True) or []
            if not v: return (None, None)
            nums0 = [int(t) for t in re.findall(r"\d+", v[0])]
            v0 = (nums0[-2], nums0[-1])
            v = cmds.polyInfo(eids[-1], ev=True) or []
            nums1 = [int(t) for t in re.findall(r"\d+", v[0])]
            v1 = (nums1[-2], nums1[-1])
            return (v0[0], v1[1]) if v0[0] != v1[1] else (v0[1], v1[0])
    except Exception:
        v = cmds.polyInfo(edge, ev=True) or []
        if not v: return (None, None)
        nums = [int(t) for t in re.findall(r"\d+", v[0])]
        return (nums[-2], nums[-1])

def _order_chain_from_edges(mesh, edges):
    if not edges: return [], False

    # 1) Graph aufbauen (nur selektierte Kanten!)
    adj = {}  # v -> set(neighbors)
    deg = {}  # v -> degree innerhalb der Auswahl
    pairs = []
    for e in edges:
        a, b = _edge_endpoints(e)
        if a is None or b is None or a == b:
            continue
        pairs.append((a, b))
        adj.setdefault(a, set()).add(b)
        adj.setdefault(b, set()).add(a)

    if not pairs:
        return [], False

    for v, ns in adj.items():
        deg[v] = len(ns)

    # 2) Endpunkte sammeln (Grad==1). Ist es ein Loop? → keine Endpunkte.
    ends = [v for v, d in deg.items() if d == 1]
    is_closed = (len(ends) == 0)

    # 3) Start bestimmen
    if not is_closed:
        # deterministisch: nimm den mit kleinster Vertex-ID (oder bbox‑left etc. – hier ID)
        start = min(ends)
    else:
        # Loop: nimm kleinste ID als Start – wird später am weitesten Paar aufgebrochen
        start = min(adj.keys())

    # 4) Pfad laufen
    ordered = [start]
    prev = None
    cur = start
    visited_edges = set()
    total_edges = len(pairs)

    while True:
        nbrs = adj.get(cur, ())
        nxt = None
        # deterministische Wahl: kleinste ID, die nicht die vorherige ist und noch nicht „benutzt“
        for n in sorted(nbrs):
            key = (min(cur, n), max(cur, n))
            if n != prev and key not in visited_edges:
                nxt = n
                visited_edges.add(key)
                break
        if nxt is None:
            break
        ordered.append(nxt)
        prev, cur = cur, nxt
        # Sicherheit: wenn wir alle Kanten einmal besucht haben, endet es
        if len(visited_edges) >= total_edges:
            break

    return ordered, is_closed

def _chain_vertices_indices(mesh, edges):
    """Gibt Vertex-Indizes in Reihenfolge zurück (z.B. [0,3,7,...]). Robust bei kleinen Lücken."""
    if not edges: return []
    pairs = [_edge_endpoints(e) for e in edges]
    a, b = pairs[0]
    chain = [a, b]
    for (ea, eb) in pairs[1:]:
        if eb == chain[-1] and ea != chain[-2]:
            chain.append(ea)
        elif ea == chain[-1] and eb != chain[-2]:
            chain.append(eb)
        elif ea == chain[0]:
            chain.insert(0, eb)
        elif eb == chain[0]:
            chain.insert(0, ea)
        else:
            if ea not in chain: chain.append(ea)
            if eb not in chain: chain.append(eb)
    out = []
    for v in chain:
        if not out or out[-1] != v:
            out.append(v)
    return out

def _ordered_vertex_components(mesh, indices):
    """Aus Indizes Vertex-Komponenten-Strings bauen."""
    return [f"{mesh}.vtx[{i}]" for i in indices]

def _vtx_world(mesh, idx):
    x, y, z = cmds.pointPosition(f"{mesh}.vtx[{idx}]", w=True)
    return om.MVector(x, y, z)

def _line_dist(p, a, b):
    ab = b - a
    if ab.length() < 1e-9: return (p - a).length()
    t = ((p - a) * ab) / (ab.length()**2)
    proj = a + t * ab
    return (p - proj).length()

def _safe_norm(v: om.MVector) -> float:
    try:
        return v.length()
    except Exception:
        return 0.0

def _angle_curvature(a: om.MVector, b: om.MVector, c: om.MVector) -> float:
    """
    Krümmung am Mittelpunkt b: 0 = gerade, 1 = sehr spitz.
    Verwendet 1 - cos(theta) (stabil und [0..2]), anschließend auf [0..1] clampen.
    """
    v1 = (a - b); v2 = (c - b)
    n1 = _safe_norm(v1); n2 = _safe_norm(v2)
    if n1 < 1e-9 or n2 < 1e-9:
        return 0.0
    v1n = v1 / n1; v2n = v2 / n2
    cos_t = max(-1.0, min(1.0, v1n * v2n))
    k = 1.0 - cos_t
    return max(0.0, min(1.0, 0.5 * k))  # 0.5 → grob in [0..1]

def _gauss_center_weight(i: int, n: int, sigma_frac: float) -> float:
    """Gauß um die Kettenmitte, 0..1."""
    if n <= 2: return 1.0
    mid = 0.5 * (n - 1)
    sigma = max(1.0, sigma_frac * (n - 1))
    t = (i - mid) / sigma
    return math.exp(-0.5 * t * t)

def _auto_apex_idx(mesh, vtx_chain):
    """
    Robuster Apex:
    - benutzt NUR Vertices der (bereits geordneten) Edge-Chain
    - Score = mix( chord_distance , curvature ) * centrality_boost
    - ignoriert Endbereiche via end_padding_frac
    """
    n = len(vtx_chain)
    if n < 3:
        return None

    # Enden als Chord-Referenz
    a = _vtx_world(mesh, vtx_chain[0])
    b = _vtx_world(mesh, vtx_chain[-1])
    ab = b - a
    ab_len = _safe_norm(ab)

    # konfig
    pad_frac  = float(OPTIONS.get("apex_end_padding_frac", 0.08))
    sigma_fac = float(OPTIONS.get("apex_centrality_sigma_frac", 0.22))
    mix       = float(OPTIONS.get("apex_mix", 0.7))  # Anteil ChordDistance

    # Index-Bereich (End-Padding wegschneiden)
    pad = max(1, int(pad_frac * n))
    lo, hi = 1, n - 2  # sowieso nicht die Enden
    lo = max(lo, pad)
    hi = min(hi, n - 1 - pad)
    if hi <= lo:
        lo, hi = 1, n - 2  # Fallback, falls zu kurze Kette

    # Positions-Cache
    pos = [_vtx_world(mesh, idx) for idx in vtx_chain]

    # 1) Chord-Distance normalisieren
    dists = []
    max_d = 1e-12
    for i in range(n):
        if ab_len < 1e-9:
            d = 0.0
        else:
            d = _line_dist(pos[i], a, b)
        dists.append(d)
        if lo <= i <= hi and d > max_d:
            max_d = d
    if max_d < 1e-9:
        max_d = 1.0
    dists_norm = [(d / max_d) for d in dists]  # 0..1

    # 2) Curvature am inneren Bereich
    curv = [0.0] * n
    max_c = 1e-12
    for i in range(1, n-1):
        c = _angle_curvature(pos[i-1], pos[i], pos[i+1])
        curv[i] = c
        if lo <= i <= hi and c > max_c:
            max_c = c
    if max_c < 1e-9:
        max_c = 1.0
    curv_norm = [(c / max_c) for c in curv]  # 0..1

    # 3) Centrality-Boost (Gauß)
    center_w = [_gauss_center_weight(i, n, sigma_fac) for i in range(n)]

    # 4) Score bauen & bestes i wählen
    best_i, best_score = None, -1.0
    for i in range(lo, hi+1):
        score_geom = mix * dists_norm[i] + (1.0 - mix) * curv_norm[i]
        score = score_geom * (0.65 + 0.35 * center_w[i])  # leichter Centrality-Boost
        if score > best_score:
            best_score, best_i = score, i

    return best_i

def _mfn_curve_from_transform(curve_xform):
    shp = (cmds.listRelatives(curve_xform, s=True, pa=True) or [None])[0]
    if not shp: return None
    sl = om.MSelectionList(); sl.add(shp)
    dag = sl.getDagPath(0)
    return om.MFnNurbsCurve(dag)

# ---------- Preview-Helfer (neu) ----------
def _moving_average(points, win):
    """Sanftes Glätten (Gleitfenster, odd)."""
    if win <= 1 or len(points) < 3:
        return points[:]
    out = []
    n = len(points)
    hw = win // 2
    for i in range(n):
        s = om.MVector(0,0,0); c = 0
        for j in range(max(0, i-hw), min(n, i+hw+1)):
            s += points[j]; c += 1
        out.append(s * (1.0/max(1,c)))
    return out

def _resample_polyline(points, target_count):
    """Gleichabständig entlang Arc-Length resamplen (inkl. erstes/letztes)."""
    n = len(points)
    if n < 2 or target_count <= 2:
        return points[:]
    # kumulative Längen
    cum = [0.0]
    for i in range(1, n):
        cum.append(cum[-1] + (points[i]-points[i-1]).length())
    total = cum[-1] or 1.0
    step = total / float(target_count-1)
    out = []
    i = 0
    for k in range(target_count):
        want = step*k
        while i < n-2 and cum[i+1] < want:
            i += 1
        seg = max(1e-12, (cum[i+1]-cum[i]))
        a = points[i]; b = points[i+1]
        u = (want - cum[i]) / seg
        out.append(a*(1.0-u) + b*u)
    return out

def _insert_corner_samples(ep, apex_idx, extra):
    """Fügt um den 'apex_idx' herum zusätzliche Punkte ein (Corner-Preserve)."""
    if extra <= 0 or apex_idx is None or len(ep) < 5:
        return ep
    out = []
    n = len(ep)
    for i in range(n-1):
        out.append(ep[i])
        if i == max(1, apex_idx-1):
            out.append(ep[i] * 0.66 + ep[i+1] * 0.34)
        if i == apex_idx:
            out.append(ep[i] * 0.34 + ep[i+1] * 0.66)
    out.append(ep[-1])
    return out

# ---------- Core ----------
def preview_curve(ctx=None, **opts):
    _apply_options(opts)
    _remember_edges_from_selection()
    edges = STATE["edges"]
    if not edges:
        _log("Select Edges.")
        return False

    mesh = edges[0].split(".e[",1)[0]
    v_idx_chain_raw, is_closed = _order_chain_from_edges(mesh, edges)
    if len(v_idx_chain_raw) < 3:
        _log("Zu kurze Edge-Kette.")
        return False

    start_idx, end_idx, v_idx_chain = _maybe_break_closed_chain(mesh, v_idx_chain_raw, is_closed)
    STATE["start"] = f"{mesh}.vtx[{start_idx}]"
    STATE["end"]   = f"{mesh}.vtx[{end_idx}]"

    # Apex bestimmen (wie gehabt) ...
    apex_i = None
    if OPTIONS.get("auto_apex", True) or not STATE.get("mid"):
        apex_i = _auto_apex_idx(mesh, v_idx_chain)
        if apex_i is None:
            _log("Apex auto failed – fahre ohne Corner-Boost fort.")
        STATE["mid"] = f"{mesh}.vtx[{apex_i}]" if apex_i is not None else None
    else:
        try:    apex_i = int(STATE["mid"].split("[",1)[1].split("]",1)[0])
        except: apex_i = None

    # WICHTIG: Original-Kettenpunkte + kumulative Längen VOR jedem Move sichern
    chain_pos = [_vtx_world(mesh, i) for i in v_idx_chain]
    cum, total = _cum_lengths_from_positions(chain_pos)
    STATE["orig_cumlen"]  = cum
    STATE["orig_totallen"] = total

    # Glätten / Resample / Corner wie gehabt ...
    frac = float(OPTIONS.get("smooth_window_frac", 0.08))
    win  = max(1, int(len(chain_pos) * frac) | 1)
    smooth = _moving_average(chain_pos, win)

    tgt = int(OPTIONS.get("preview_points", 32))
    tgt = max(int(OPTIONS.get("min_preview_points", 6)), min(int(OPTIONS.get("max_preview_points", 64)), tgt))
    tgt = min(tgt, max(6, len(smooth)))
    ep = _resample_polyline(smooth, tgt)

    if float(OPTIONS.get("corner_boost", 0.0)) > 0.0 and apex_i is not None:
        apex_pos = _vtx_world(mesh, v_idx_chain[apex_i])
        nearest, bestd = 0, 1e18
        for i, p in enumerate(ep):
            d = (p - apex_pos).length()
            if d < bestd: bestd = d; nearest = i
        extra = 1 if OPTIONS["corner_boost"] < 0.5 else 2
        ep = _insert_corner_samples(ep, nearest, extra)

    # alte VDPRO-Kurve löschen + neue erzeugen (unverändert) ...
    if STATE.get("curve") and _is_vdpro_curve(STATE["curve"]):
        try: cmds.delete(STATE["curve"])
        except: pass

    name = _unique_curve_name()
    pts = [(p.x, p.y, p.z) for p in ep]
    deg = max(1, min(3, int(OPTIONS.get("preview_degree", 3))))
    crv = cmds.curve(d=deg, ep=pts, name=name)
    try:
        cmds.rebuildCurve(crv, ch=False, rpo=True, rt=0, end=1, kr=0, kcp=False, kep=True,
                          kt=False, s=max(0, len(pts)-deg-1), d=deg, tol=0.0001, keepRange=2)
    except Exception:
        pass
    _ensure_tag_attr(crv)
    try:
        cmds.setAttr(crv + ".overrideEnabled", 1)
        cmds.setAttr(crv + ".overrideColor", 13)
    except Exception:
        pass

    STATE["curve"] = crv
    STATE["last_mesh"] = mesh
    STATE["ordered"] = _ordered_vertex_components(mesh, v_idx_chain)

    _sel_edges(edges)

    # Auto-Distribute wie bisher – nutzt jetzt (falls gesetzt) die orig_cumlen. :contentReference[oaicite:1]{index=1}
    try:
        ok = distribute(ctx)
        if not ok:
            _log("Auto-Distribute nach Preview: kein Erfolg (siehe Log).")
    except Exception as e:
        _log(f"Auto-Distribute nach Preview fehlgeschlagen: {e}")
    return True

def distribute(ctx=None, **opts):
    _apply_options(opts)
    ordered = STATE.get("ordered") or []
    crv = STATE.get("curve")

    if not ordered:
        _log("Keine gecachten Vertices gefunden. Bitte zuerst PreviewCurve ausführen.")
        return False
    if not (crv and cmds.objExists(crv) and _is_vdpro_curve(crv)):
        curvs = _list_vdpro_curves()
        crv = curvs[-1] if curvs else None
    if not crv:
        _log("Keine VDPRO-Kurve gefunden.")
        return False

    try:
        fn = _mfn_curve_from_transform(crv)
        if not fn:
            _log("Ungültige Kurve.")
            return False

        Lc = fn.length()
        cnt = len(ordered)
        if cnt < 2:
            _log("Not enough verts in STATE['ordered'].")
            return False

        use_even = bool(OPTIONS.get("even_distribution", True))
        cum = STATE.get("orig_cumlen")
        L0  = STATE.get("orig_totallen", None)

        # Fallback, wenn keine Originaldaten oder Größenabweichung
        if (not use_even) and (isinstance(cum, list)) and (isinstance(L0, (int, float))) and (len(cum) == cnt):
            scale = (Lc / L0) if L0 and L0 > 1e-9 else 1.0
            with _undo_chunk("VDPRO Distribute (preserve spacing)"):
                for i, vtx in enumerate(ordered):
                    dist = float(cum[i]) * scale
                    u = fn.findParamFromLength(max(0.0, min(Lc, dist)))
                    pos = fn.getPointAtParam(u, om.MSpace.kWorld)
                    cmds.xform(vtx, ws=True, t=(pos.x, pos.y, pos.z))
        else:
            step = Lc / float(cnt - 1)
            with _undo_chunk("VDPRO Distribute (even)"):
                for i, vtx in enumerate(ordered):
                    dist = step * i
                    u = fn.findParamFromLength(dist)
                    pos = fn.getPointAtParam(u, om.MSpace.kWorld)
                    cmds.xform(vtx, ws=True, t=(pos.x, pos.y, pos.z))

        _log("Distributed.")
        return True
    except Exception as e:
        _log(f"Distribution failed: {e}")
        return False


def _set_pivot_world(xform, p):
    """Setzt Rotate- und Scale-Pivot im Worldspace auf p."""
    try:
        cmds.xform(xform, ws=True, piv=(p.x, p.y, p.z))
    except Exception:
        pass

def _curve_start_end(fn):
    """(p0, p1) der Kurve im Worldspace."""
    u0, u1 = fn.knotDomain
    return (
        fn.getPointAtParam(u0, om.MSpace.kWorld),
        fn.getPointAtParam(u1, om.MSpace.kWorld),
    )

def reuse_curve(ctx=None, **opts):
    _apply_options(opts)
    _remember_edges_from_selection()
    edges = STATE["edges"]
    if not edges:
        _log("Bitte neue Edges selektieren.")
        return False

    mesh = edges[0].split(".e[",1)[0]

    # robuste Kettenableitung (bestehend) ...
    try:
        v_idx_chain_raw, is_closed = _order_chain_from_edges(mesh, edges)
    except Exception:
        v_idx_chain_raw = _chain_vertices_indices(mesh, edges)
        is_closed = False
    if len(v_idx_chain_raw) < 2:
        _log("Zu kurze Kette.")
        return False

    start_idx, end_idx, v_idx_chain = _maybe_break_closed_chain(mesh, v_idx_chain_raw, is_closed)

    # Anchor-Logik (bestehend) ...
    try:
        first_e = (cmds.filterExpand(edges, sm=32) or [edges[0]])[0]
        ea, eb = _edge_endpoints(first_e)
        pa = _vtx_world(mesh, ea); pb = _vtx_world(mesh, eb)
        a = _vtx_world(mesh, start_idx); b = _vtx_world(mesh, end_idx)
        d_start = min((a - pa).length(), (a - pb).length())
        d_end   = min((b - pa).length(), (b - pb).length())
        if d_end + 1e-8 < d_start:
            start_idx, end_idx = end_idx, start_idx
            v_idx_chain = list(reversed(v_idx_chain))
    except Exception:
        pass

    # State + NEU: Original-Längen der NEUEN Kette sichern (vor Dist!)
    STATE["start"]      = f"{mesh}.vtx[{start_idx}]"
    STATE["end"]        = f"{mesh}.vtx[{end_idx}]"
    STATE["ordered"]    = _ordered_vertex_components(mesh, v_idx_chain)
    STATE["last_mesh"]  = mesh

    chain_pos = [_vtx_world(mesh, i) for i in v_idx_chain]
    cum, total = _cum_lengths_from_positions(chain_pos)
    STATE["orig_cumlen"]   = cum
    STATE["orig_totallen"] = total

    # Kurve ausrichten (bestehend) ...
    crv = STATE.get("curve")
    if not (crv and cmds.objExists(crv) and _is_vdpro_curve(crv)):
        curvs = _list_vdpro_curves()
        crv = curvs[-1] if curvs else None
    if not crv:
        _log("Keine VDPRO-Kurve im State.")
        return False

    fn = _mfn_curve_from_transform(crv)
    if not fn:
        _log("Ungültige Kurve.")
        return False

    a_new = _vtx_world(mesh, start_idx)
    b_new = _vtx_world(mesh, end_idx)
    v_new = b_new - a_new
    len_new = v_new.length()
    if len_new < 1e-8:
        _log("Neue Kette degeneriert (Start~End).")
        return False

    u0, u1 = fn.knotDomain
    p0 = fn.getPointAtParam(u0, om.MSpace.kWorld)
    p1 = fn.getPointAtParam(u1, om.MSpace.kWorld)
    v_old = om.MVector(p1) - om.MVector(p0)
    len_old = v_old.length()
    if len_old < 1e-8:
        _log("Alte Kurve degeneriert (Start~End).")
        return False

    cmds.xform(crv, ws=True, piv=(p0.x, p0.y, p0.z))
    try:
        v_old_n = v_old / len_old
        v_new_n = v_new / len_new
        q = om.MQuaternion(v_old_n, v_new_n)
        eul = q.asEulerRotation()
        cmds.rotate(math.degrees(eul.x), math.degrees(eul.y), math.degrees(eul.z), crv, r=True, ws=True)
    except Exception as e:
        _log(f"Rotation fehlgeschlagen: {e}")
        return False

    try:
        s = len_new / len_old
        cmds.scale(s, s, s, crv, r=True, ws=True, p=(p0.x, p0.y, p0.z))
    except Exception as e:
        _log(f"Scale fehlgeschlagen: {e}")
        return False

    try:
        fn = _mfn_curve_from_transform(crv)
        p0_now = fn.getPointAtParam(fn.knotDomain[0], om.MSpace.kWorld)
        delta = om.MVector(a_new) - om.MVector(p0_now)
        cmds.xform(crv, ws=True, t=(delta.x, delta.y, delta.z), r=True)
    except Exception as e:
        _log(f"Translate fehlgeschlagen: {e}")
        return False

    try:
        flipped_curve = _flip_if_better(crv, om.MVector(a_new), om.MVector(b_new))
    except Exception as e:
        _log(f"Richtungsprüfung fehlgeschlagen: {e}")
        flipped_curve = False

    try:
        fn = _mfn_curve_from_transform(crv)
        reversed_order = _maybe_reverse_vertex_order_for_better_fit(fn)
    except Exception as e:
        _log(f"Order-Fit-Test fehlgeschlagen: {e}")
        reversed_order = False

    STATE["curve"] = crv
    _sel_edges(edges)

    _log("ReUse: Kurve ausgerichtet"
         + (" (curve flipped)" if flipped_curve else "")
         + (" (order reversed)" if reversed_order else "")
         + ", verteile …")
    return distribute(ctx)


# ---------- RESET ----------
def reset_all(ctx=None, **opts):
    _apply_options(opts)  # ignoriert unbekannte keys
    keep = bool(OPTIONS.get("keep_curves_on_reset", False))

    # VDPRO-Kurven löschen (getaggt ODER Namen mit Prefix)
    if not keep:
        all_xf = cmds.ls(type="transform") or []
        for t in all_xf:
            try:
                if _is_vdpro_curve(t) or t.startswith(CURVE_PREFIX):
                    cmds.delete(t)
            except Exception:
                pass

    # State leeren
    STATE.update({"edges": [], "start": None, "mid": None, "end": None,
                  "curve": None, "last_mesh": None, "ordered": []})
    try:
        cmds.select(clear=True)
    except Exception:
        pass
    _log("Reset done.")



# ---------- Runner Entry ----------
def run(ctx=None):
    return {
        "actions": {
            "preview_curve": preview_curve,
            "distribute":    distribute,
            "reuse_curve":   reuse_curve,
            "simplify_curve": simplify_curve,
            "complexify_curve": complexify_curve,
            "reset_all":     reset_all,
        },
        "set_option": set_option
    }
