# actions.py — Business-Logik & State für den ZenTools Hub
from __future__ import annotations
import os, json, time, traceback, inspect, shutil
from typing import Any, Callable, Dict, Optional, Tuple
import urllib.parse

# --- Qt optional (nur für evtl. spätere Erweiterungen; keine UI-Elemente nutzen!)
try:
    from pjots_zen_tools.core import qt_compat as QC  # noqa
except Exception:
    QC = None  # noqa

# Projekt-APIs
try:
    from pjots_zen_tools.hub.registry import get_tools
    from pjots_zen_tools.hub.discovery import discover_and_register
except Exception:
    def get_tools() -> Dict[str, Dict[str, Any]]: return {}
    def discover_and_register() -> None: return None

# Updater / Shelves
try:
    from pjots_zen_tools.core import updater
except Exception:
    updater = None

try:
    from pjots_zen_tools.core import shelves as shelves_api
except Exception:
    shelves_api = None

# Ressourcen (für Shelf-Icons)
try:
    from pjots_zen_tools import resources as R
except Exception:
    R = None  # type: ignore

# ---------------------------------------------------------------------
# Konstanten (Single Source)
# ---------------------------------------------------------------------
DEFAULT_MANIFEST = os.environ.get(
    "ZEN_TOOLS_MANIFEST", "https://get.pjotszen.tools/manifest.json"
)

ENTITLEMENTS_URL = os.environ.get(
    "ZEN_TOOLS_ENTITLEMENTS", "https://get.pjotszen.tools/entitlements_check.php"
)

LEAD_ENDPOINT = os.environ.get(
    "ZEN_TOOLS_LEAD_ENDPOINT", "https://get.pjotszen.tools/app/save_lead_v2.php"
)

REQUEST_ENDPOINT = os.environ.get(
    "ZEN_TOOLS_REQUEST_ENDPOINT", "https://get.pjotszen.tools/app/save_request_v1.php"
)

# ---------------------------------------------------------------------
# Files / IO
# ---------------------------------------------------------------------
def _user_prefs_dir():
    try:
        import maya.cmds as _cmds  # type: ignore
        return _cmds.internalVar(userPrefDir=True).replace("\\", "/")
    except Exception:
        return os.path.join(os.path.expanduser("~"), ".pjots_zen").replace("\\", "/")

LICENSE_PATH = os.path.join(_user_prefs_dir(), "pjots_zen_tools_license.json").replace("\\", "/")
ENTITLE_CACHE_PATH = os.path.join(_user_prefs_dir(), "pjots_zen_tools_entitlements_cache.json").replace("\\", "/")
REQUEST_OUTBOX_DIR = os.path.join(_user_prefs_dir(), "requests_outbox").replace("\\", "/")

def _ensure_outbox_dir():
    os.makedirs(REQUEST_OUTBOX_DIR, exist_ok=True); return REQUEST_OUTBOX_DIR

def _ensure_dir_for(p):
    d = os.path.dirname(p)
    if d and not os.path.exists(d):
        os.makedirs(d, exist_ok=True)

def _load_json_safe(p, default=None):
    try:
        if os.path.exists(p):
            with open(p, "r", encoding="utf-8") as f:
                return json.load(f)
    except Exception:
        pass
    return default

def _save_json_safe(p, data):
    try:
        _ensure_dir_for(p)
        with open(p, "w", encoding="utf-8") as f:
            json.dump(data or {}, f, indent=2, ensure_ascii=False)
        return True
    except Exception:
        return False

def _current_hub_dir() -> str:
    """Gibt den Ordner zurück, in dem dieses actions.py aktuell liegt."""
    return os.path.abspath(os.path.dirname(__file__))

def _compute_tool_dir_for(module_id: str) -> str:
    """
    Standard-Zielordner für normale Tools:
    <pjots_zen_tools>/tools/<module_id>
    """
    this_dir = os.path.abspath(os.path.dirname(__file__))      # z.B. .../pjots_zen_tools/hub
    pjots_root = os.path.dirname(this_dir)                     # .../pjots_zen_tools
    tools_root = os.path.join(pjots_root, "tools")
    return os.path.join(tools_root, module_id)

    # Prefer cached self.email; fallback to license profile
    email = (self.email or self.get_profile().get("email") or "").strip().lower()
    if not email:
        return base

    parts = urllib.parse.urlsplit(base)
    qs = dict(urllib.parse.parse_qsl(parts.query, keep_blank_values=True))
    qs["email"] = email
    new_parts = parts._replace(query=urllib.parse.urlencode(qs))
    return urllib.parse.urlunsplit(new_parts)
import shutil

def _postfix_relocate_hub() -> None:
    src_tmp = _compute_tool_dir_for("zen_hub")   # wohin der Installer immer schreibt
    dst_real = _current_hub_dir()                # von wo unser Hub gerade läuft

    # Normalfall: wenn src_tmp == dst_real, müssen wir nix tun.
    if os.path.normpath(src_tmp) == os.path.normpath(dst_real):
        return

    # Wenn es die tmp-Version überhaupt nicht gibt, auch nix tun.
    if not os.path.isdir(src_tmp):
        return

    # 1. rüberkopieren / überschreiben
    # Wir gehen Datei für Datei durch, um nicht versehentlich den Zielordner zu löschen.
    for root, dirs, files in os.walk(src_tmp):
        rel = os.path.relpath(root, src_tmp)
        target_root = os.path.join(dst_real, rel) if rel != "." else dst_real
        if not os.path.isdir(target_root):
            os.makedirs(target_root, exist_ok=True)
        # Kopiere Dateien
        for f in files:
            src_f = os.path.join(root, f)
            dst_f = os.path.join(target_root, f)
            shutil.copy2(src_f, dst_f)

    # 2. tmp-Kopie wieder entfernen, damit wir nicht 2 Hubs sehen
    try:
        shutil.rmtree(src_tmp)
    except Exception as e:
        # Nicht kritisch, aber loggen wär nett
        pass


# ---------------------------------------------------------------------
# Entitlements / Lead / License
# ---------------------------------------------------------------------
def _user_rec(ent: dict, email: str) -> dict:
    users = (ent or {}).get("users") or {}
    return users.get((email or "").strip().lower(), {}) or {}

def _is_commercial_active_for(email: str, ent: dict) -> bool:
    email = (email or "").strip().lower()
    if not email:
        return False
    u = _user_rec(ent, email)
    c = (u.get("commercial") or {})
    if c.get("status") in ("active", "grace"):
        return True
    # studio seat inheritance
    users = (ent or {}).get("users") or {}
    for rec in users.values():
        cc = (rec.get("commercial") or {})
        if cc.get("status") in ("active", "grace") and email in (cc.get("seat_assignments") or []):
            return True
    return False

def _append_email_qs(base: str, email: Optional[str]) -> str:
    email = (email or "").strip().lower()
    if not email:
        return base
    parts = urllib.parse.urlsplit(base)
    qs = dict(urllib.parse.parse_qsl(parts.query, keep_blank_values=True))
    qs["email"] = email
    new_parts = parts._replace(query=urllib.parse.urlencode(qs))
    return urllib.parse.urlunsplit(new_parts)

def _per_tool_grants(email: str, ent: dict):
    u = _user_rec(ent, email)
    tools = (u.get("tools") or {})
    # trust server; ignore expires_at parsing here; hub accepts server's word
    return set(tools.keys())

def _read_free_baseline_from_v1(ent: dict):
    allow = (ent or {}).get("allow") or {}
    base = set(allow.get("*") or [])
    return base

def _compute_allowed_ids(email: str, ent: dict, catalog_ids: set, free_baseline: set):
    # commercial (active/grace) unlocks everything in catalog
    if _is_commercial_active_for(email, ent):
        return set(catalog_ids)
    # otherwise: free baseline (v1 *) + per-tool micro-tx grants
    grants = _per_tool_grants(email, ent)
    return set(free_baseline) | set(grants)

def _normalize_entitlements_shape(raw: dict, email: Optional[str]) -> dict:
    email = (email or "").strip().lower()
    if not raw or not isinstance(raw, dict):
        return {"users": {}, "allow": {"*": []}}

    # Already v2 global?
    if "users" in raw:
        # ensure allow exists
        if "allow" not in raw or not isinstance(raw.get("allow"), dict):
            raw = dict(raw)
            raw["allow"] = {"*": []}
        return raw

    # Single-user view → wrap
    commercial = raw.get("commercial") or {}
    tools = raw.get("tools") or {}
    allow = raw.get("allow") or {"*": []}  # optional from server
    users = {}
    if email:
        users[email] = {"commercial": commercial, "tools": tools}
    return {"users": users, "allow": allow}

def _load_license():
    lic = _load_json_safe(LICENSE_PATH, {}) or {}
    return lic if isinstance(lic, dict) else {}

def _save_license(data):
    try:
        if isinstance(data, str):
            data = {"email": (data or "").strip().lower()}
        else:
            data = dict(data or {})
            data["email"] = (data.get("email", "") or "").strip().lower()
        if not data.get("email"):
            return False
        return _save_json_safe(LICENSE_PATH, data)
    except Exception:
        return False

def _fetch_entitlements_online(url: str):
    try:
        import urllib.request, ssl
        ctx = ssl.create_default_context()
        req = urllib.request.Request(url, headers={"User-Agent": "PjotsZenHub/1.0"})
        with urllib.request.urlopen(req, timeout=12, context=ctx) as resp:
            return json.loads(resp.read().decode("utf-8")) or {}
    except Exception:
        return {}

def _load_entitlements_cached(ttl_seconds=86400, email: Optional[str] = None):
    """
    Cache is keyed by email to avoid mixing different users' views.
    """
    email = (email or "").strip().lower()
    now = time.time()
    cache = _load_json_safe(ENTITLE_CACHE_PATH, {}) or {}

    # If cache email differs or expired → fetch
    if cache.get("_ts") and cache.get("_email") == email and (now - float(cache.get("_ts", 0))) < ttl_seconds:
        return cache.get("data") or {}

    # Build URL with ?email=...
    url = _append_email_qs(ENTITLEMENTS_URL, email)
    raw = _fetch_entitlements_online(url)
    data = _normalize_entitlements_shape(raw, email)

    _save_json_safe(ENTITLE_CACHE_PATH, {"_ts": now, "_email": email, "data": data})
    return data or {}


def _entitled_set(email: Optional[str], entitlements: dict):
    allow = (entitlements or {}).get("allow", {})
    base = set(allow.get("*", []) or [])
    if email and email in allow:
        base.update(allow[email] or [])
    return base

def _submit_lead_email(email: str, name: str = "", studio: bool = False, marketing: bool = False, timeout_s: int = 8):
    email = (email or "").strip().lower()
    if not email or "@" not in email:
        return False, "E-mail not valid."
    try:
        import urllib.request, ssl, json as _json
        ctx = ssl.create_default_context()
        payload = _json.dumps({"email": email, "name": name or "", "studio": bool(studio), "marketing": bool(marketing)}).encode("utf-8")
        req = urllib.request.Request(LEAD_ENDPOINT, data=payload, method="POST",
            headers={"Content-Type":"application/json","User-Agent":"PjotsZenHub/1.0"})
        with urllib.request.urlopen(req, timeout=timeout_s, context=ctx) as resp:
            data = _json.loads(resp.read().decode("utf-8") or "{}")
            ok = bool(data.get("ok"))
            return (True, "Confirmation email sent.") if ok else (False, "Server answered (not ok).")
    except Exception as ex:
        return False, f"Network error: {ex}"

def get_license_snapshot(ent_ttl: int = 86400) -> dict:
    """
    Lightweight helper for tools (ProTools, etc.) to query the current licensing state.

    Uses only the local license + entitlements cache and refreshes entitlements
    when the TTL has expired.

    Returns a dict:
        {
            "email": str or None,
            "commercial_active": bool,
        }
    """
    lic = _load_license() or {}
    email = (lic.get("email") or "").strip().lower()

    # Holt Entitlements aus Cache (inkl. Refresh nach TTL)
    ent = _load_entitlements_cached(ttl_seconds=ent_ttl, email=email)

    # Zentrale Logik: "commercial: 'active'" => True
    commercial = False
    try:
        commercial = bool(_is_commercial_active_for(email, ent))
    except Exception:
        commercial = False

    return {
        "email": email or None,
        "commercial_active": commercial,
    }

# ---------------------------------------------------------------------
# Manifest / Merge / Versions
# ---------------------------------------------------------------------
def _semver_cmp(a: str, b: str) -> int:
    def norm(v):
        v = (v or "0").replace("-", ".")
        return [int(x) if x.isdigit() else x for x in v.split(".")]
    na, nb = norm(a), norm(b)
    for xa, xb in zip(na, nb):
        if xa == xb: continue
        return -1 if xa < xb else 1
    return (len(na) > len(nb)) - (len(na) < len(nb))

def _load_manifest(url: str = DEFAULT_MANIFEST) -> dict:
    try:
        import urllib.request
        with urllib.request.urlopen(url, timeout=12) as resp:
            return json.loads(resp.read().decode("utf-8")) or {}
    except Exception:
        return {"tools": []}

def _merge_catalog(local_reg: dict, manifest: dict) -> dict:
    merged: Dict[str, Dict[str, Any]] = {}
    for mid, meta in (local_reg or {}).items():
        m = dict(meta)
        m["installed"] = True
        m["remote_version"] = None
        m["manifest_entry"] = None
        merged[mid] = m
    for t in (manifest or {}).get("tools", []):
        mid = t.get("id")
        if not mid:
            continue
        blob = {
            "title": t.get("title", mid),
            "description": t.get("description", ""),
            "tags": t.get("tags", []),
            "icon": t.get("icon"),
            "version": "0.0.0",
            "launcher": None,
        }
        m = merged.get(mid, blob)
        m["remote_version"] = t.get("version", "0.0.0")
        m["manifest_entry"] = t
        if mid in merged:
            mloc = merged[mid]
            for k in ("title", "description", "tags", "icon"):
                mloc[k] = mloc.get(k) or m.get(k)
            mloc["remote_version"] = m["remote_version"]
            mloc["manifest_entry"] = t
        else:
            m["installed"] = False
            merged[mid] = m
    return merged

def _make_update_item(module_id: str, meta: dict) -> dict:
    man = meta.get("manifest_entry") or {}
    return {
        "module_id": module_id,
        "remote_version": man.get("version", meta.get("remote_version") or "0.0.0"),
        "zip_url": man.get("zip_url"),
        "sha256": man.get("sha256"),
        "entry": man.get("entry"),
        "title": meta.get("title", module_id),
        "description": meta.get("description", ""),
        "icon": meta.get("icon"),
        "tags": meta.get("tags", []),
    }

# ---------------------------------------------------------------------
# HubActions (Stateful) — UI delegiert an diese Klasse
# ---------------------------------------------------------------------
class HubActions:
    def __init__(self, log: Optional[Callable[[str, str], None]] = None):
        self._log_cb = log or (lambda msg, lvl="INFO": None)

        # Maya cmds optional
        try:
            import maya.cmds as cmds  # type: ignore
            self.cmds = cmds
        except Exception:
            self.cmds = None

        # State
        self.email: Optional[str] = None
        self.entitlements: dict = {}
        self.allowed_ids: set[str] = set()
        self.catalog_cache: Dict[str, Dict[str, Any]] = {}
        self.updates_index: Dict[str, Dict[str, Any]] = {}

        # Initial load
        self.reload_all()

    # ---------- Logging ----------
    def _log(self, msg: str, level: str = "INFO") -> None:
        try:
            self._log_cb(msg, level)
        except Exception:
            pass

    # ---------- Loads / Calculations ----------
    def get_profile(self) -> dict:
        lic = _load_license()
        # Fallbacks, falls ältere Lizenzdatei nur die E-Mail enthält
        return {
            "email": (lic.get("email") or "").strip().lower(),
            "name": lic.get("name", ""),
            "studio": bool(lic.get("studio", False)),
            "marketing": bool(lic.get("marketing", False)),
        }

    def get_commercial_url(self) -> str:
        """
        Build the paywall URL and append the current email as ?email=...
        Uses GET_COMMERCIAL_URL env or defaults to the production URL.
        """
        base = os.getenv(
            "GET_COMMERCIAL_URL",
            "https://get.pjotszen.tools/app/get_commercial.php"
        )

        email = (self.email or "").strip().lower()
        if not email:
            return base

        parts = urllib.parse.urlsplit(base)
        qs = dict(urllib.parse.parse_qsl(parts.query, keep_blank_values=True))
        qs["email"] = email
        new_parts = parts._replace(query=urllib.parse.urlencode(qs))
        return urllib.parse.urlunsplit(new_parts)

    def _gather_sysinfo(self) -> dict:
        si = {
            "hub_version": "",
            "maya_version": "",
            "os": "",
            "commercial_active": bool(getattr(self, "_commercial_active", False)),
            "timestamp": int(time.time()),
        }
        # hub version: best-effort from hub entry in catalog
        try:
            hub_meta = (self.catalog_cache or {}).get("zen_hub", {}) or {}
            si["hub_version"] = hub_meta.get("version", "") or ""
        except Exception:
            pass
        # maya version
        try:
            if self.cmds:
                si["maya_version"] = str(self.cmds.about(version=True))
        except Exception:
            pass
        # OS
        try:
            import platform
            si["os"] = f"{platform.system()} {platform.release()}"
        except Exception:
            pass
        return si

    def _outbox_enqueue(payload: dict) -> str | None:
        try:
            _ensure_outbox_dir()
            fn = f"req_{int(time.time())}.json"
            path = os.path.join(REQUEST_OUTBOX_DIR, fn)
            with open(path, "w", encoding="utf-8") as f:
                json.dump(payload, f, ensure_ascii=False, indent=2)
            return path
        except Exception:
            return None

    def retry_outbox(self, timeout_s: int = 8) -> int:
        """Try sending all pending outbox JSON files. Returns count of sent."""
        sent = 0
        _ensure_outbox_dir()
        files = sorted([f for f in os.listdir(REQUEST_OUTBOX_DIR) if f.endswith(".json")])
        for fn in files:
            p = os.path.join(REQUEST_OUTBOX_DIR, fn)
            try:
                with open(p, "r", encoding="utf-8") as f:
                    payload = json.load(f) or {}
                ok, _ = self._post_request(payload, timeout_s=timeout_s)
                if ok:
                    try: os.remove(p)
                    except Exception: pass
                    sent += 1
            except Exception:
                pass
        return sent

    def _post_request(self, payload: dict, timeout_s: int = 10):
        try:
            import urllib.request, ssl, json as _json
            ctx = ssl.create_default_context()
            req = urllib.request.Request(
                REQUEST_ENDPOINT,
                data=_json.dumps(payload).encode("utf-8"),
                method="POST",
                headers={"Content-Type":"application/json","User-Agent":"PjotsZenHub/1.0"}
            )
            with urllib.request.urlopen(req, timeout=timeout_s, context=ctx) as resp:
                data = _json.loads(resp.read().decode("utf-8") or "{}")
                if bool(data.get("ok")):
                    return True, data.get("id", "ok")
                return False, data.get("message", "Server answered (not ok).")
        except Exception as ex:
            return False, f"Network error: {ex}"

    def send_feedback_request(self, payload: dict) -> tuple[bool, str]:
        """UI-facing API. Enrich, enforce commercial gating for tool requests, post, fallback to outbox on failure."""
        try:
            # Normalize + enrich
            p = dict(payload or {})
            p.setdefault("type", "feedback")
            p.setdefault("subject", "")
            p.setdefault("message", "")
            p.setdefault("tool_id", "zen_hub")

            # Commercial flag (from entitlements)
            remote_active = bool(getattr(self, "_commercial_active", False))

            # If a non-commercial user somehow sends a "request", downgrade it to feedback and mark the subject
            if p.get("type") == "request" and not remote_active:
                subj = (p.get("subject") or "").strip()
                if subj:
                    subj = "[free-user request] " + subj
                else:
                    subj = "[free-user request]"
                p["subject"] = subj
                p["type"] = "feedback"

            # Ensure email
            if not p.get("email"):
                p["email"] = (self.email or self.get_profile().get("email") or "").strip().lower()

            # Optional system info
            if p.get("include_sysinfo", True):
                p["sysinfo"] = self._gather_sysinfo()

            # Attach log excerpt if requested (UI sets attach_log via checkbox; not implemented yet)
            if p.get("log_excerpt") is None and p.get("attach_log", False):
                # Placeholder: for now, nothing is added here; hub log could be attached later.
                pass

            ok, msg = self._post_request(p, timeout_s=10)
            if ok:
                return True, str(msg)

            # Fallback → Outbox (keine Änderung zum bisherigen Verhalten)
            _outbox_enqueue(p)
            return False, f"{msg} (stored to outbox)"
        except Exception as ex:
            try:
                _outbox_enqueue(payload or {})
            except Exception:
                pass
            return False, f"Exception: {ex}"


    def reload_all(self, ent_ttl: int = 86400) -> None:
        # License/Entitlements
        lic = _load_license()
        self.email = (lic or {}).get("email")
        # fetch per-email entitlements
        self.entitlements = _load_entitlements_cached(ttl_seconds=ent_ttl, email=self.email)

        # Merge local registry + manifest
        local_reg = get_tools() or {}
        manifest = _load_manifest(DEFAULT_MANIFEST)
        self.catalog_cache = _merge_catalog(local_reg, manifest)


        # Recompute entitlements using v2 schema (commercial + per-tool + studio seats)
        try:
            # free baseline (server -> allow.*), fallback to hardcoded minimal set if missing
            free_base = _read_free_baseline_from_v1(self.entitlements)
            if not free_base:
                # Fallback: minimal free set (adjust to your policy)
                free_base = {"pro_tools", "controller_library"}
            catalog_ids = set(self.catalog_cache.keys())
            self._commercial_active = _is_commercial_active_for(self.email or "", self.entitlements)
            self.allowed_ids = _compute_allowed_ids(self.email or "", self.entitlements, catalog_ids, free_base)
        except Exception as e:
            self._log(f"v2 entitlement compute failed, keep legacy set: {e}", level="WARN")
            self.allowed_ids = _entitled_set(self.email, self.entitlements)

        # Reconcile local "commercial" flag with server (downgrade local if mismatch)
        try:
            lic = _load_license()
            if lic.get("studio") and not getattr(self, "_commercial_active", False):
                lic = dict(lic)
                lic["studio"] = False
                _save_license(lic)
                self._log("Local commercial flag disabled due to server mismatch.", level="WARN")
        except Exception as e:
            self._log(f"Failed to reconcile local license: {e}", level="WARN")

        # Build updates index
        self.updates_index.clear()
        for mid, meta in self.catalog_cache.items():
            rv = meta.get("remote_version")
            lv = meta.get("version", "0.0.0")

            entitled = (mid in self.allowed_ids)
            if not self.email:
                # unregistered → nur freie Tools („*“)
                entitled = (mid in self.allowed_ids)

            if rv and _semver_cmp(lv, rv) < 0 and entitled and meta.get("installed"):
                self.updates_index[mid] = _make_update_item(mid, meta)

        self._log(
            f"Catalog loaded: {len(self.catalog_cache)} tools, "
            f"{len(self.updates_index)} updates, allowed={len(self.allowed_ids)} for {self.email or 'UNREGISTERED'}"
        )

    def rescan(self) -> None:
        self._log("Rescan tools...")
        discover_and_register()
        self.reload_all()

    # ---------- Registration / Leads ----------
    def register_email(self, email: str, name: str = "", studio: bool = False, marketing: bool = False):
        email = (email or "").strip().lower()
        if not email or "@" not in email:
            return False, "Please enter a valid email."
        if not _save_license({"email": email, "name": name, "studio": bool(studio), "marketing": bool(marketing)}):
            return False, "Could not save registration locally."

        # Lead minimal senden (nur Email zwingend – Server kann später erweitert werden)
        ok_lead, msg_lead = _submit_lead_email(email, name, studio, marketing)
        self._log(f"Lead submit result: ok={ok_lead}, msg={msg_lead}")

        # Entitlements-Cache invalidieren und alles neu laden
        try:
            if os.path.exists(ENTITLE_CACHE_PATH):
                os.remove(ENTITLE_CACHE_PATH)
        except Exception:
            pass
        self.reload_all(ent_ttl=0)
        return True, f"Registered as {email}."


    def logout(self) -> bool:
        try:
            if os.path.exists(LICENSE_PATH):
                os.remove(LICENSE_PATH)
            if os.path.exists(ENTITLE_CACHE_PATH):
                os.remove(ENTITLE_CACHE_PATH)
        except Exception as e:
            self._log(f"Logout file cleanup error: {e}", level="WARN")

        self.email = None
        self.entitlements = {}
        self.allowed_ids = set()
        self.reload_all(ent_ttl=0)
        return True

    # ---------- Updates ----------
    def force_check_updates(self) -> int:
        # Entitlements hart aktualisieren
        try:
            if os.path.exists(ENTITLE_CACHE_PATH):
                os.remove(ENTITLE_CACHE_PATH)
        except Exception:
            pass
        if updater is None:
            self._log("Updater missing for check_updates", level="WARN")
        self.reload_all(ent_ttl=0)
        # updates_index wurde im reload_all aufgebaut
        return len(self.updates_index)

    # ---------- Install / Update / Uninstall ----------
    def update_selected(self, module_id: str) -> Tuple[bool, str]:
        is_hub = (module_id == "zen_hub")

        if (not is_hub) and (module_id not in self.allowed_ids):
            return False, "Not entitled for this tool."

        upd = self.updates_index.get(module_id)
        if not upd:
            meta = self.catalog_cache.get(module_id, {}) or {}
            if not meta.get("remote_version"):
                return False, "No update available."
            upd = _make_update_item(module_id, meta)

        upd = dict(upd)
        if not upd.get("zip_url"):
            return False, "No download URL in manifest."

        try:
            if updater is None:
                raise RuntimeError("Installer-Module not found (core.updater).")

            # rein informatives Log – wir wissen, dass updater uns ignoriert,
            # aber fürs Debug behalten wir das drinnen:
            if is_hub:
                self._log(
                    "download_and_install for %s (hub self-update) → %s"
                    % (module_id, _current_hub_dir())
                )
            else:
                self._log(
                    "download_and_install for %s → %s"
                    % (module_id, _compute_tool_dir_for(module_id))
                )

            # call installer
            try:
                msg = updater.download_and_install(upd, manifest_url=DEFAULT_MANIFEST)
            except TypeError:
                msg = updater.download_and_install(upd)

            # wenn hub: nachträglich an richtigen ort verschieben + tmp löschen
            if is_hub:
                _postfix_relocate_hub()

            self._log(f"Update finished for {module_id}")

            self.rescan()
            return True, (msg or "Updated.")
        except Exception as e:
            self._log(f"Update error for {module_id}: {e}", level="ERROR")
            traceback.print_exc()
            return False, str(e)

    def install_selected(self, module_id: str) -> Tuple[bool, str]:
        is_hub = (module_id == "zen_hub")

        meta = self.catalog_cache.get(module_id, {}) or {}

        if (not is_hub) and (module_id not in self.allowed_ids):
            return False, "Not entitled for this tool."

        if meta.get("installed"):
            return True, "Already installed."

        upd = _make_update_item(module_id, meta)
        if not upd.get("zip_url"):
            return False, "No download URL in manifest."

        try:
            if updater is None:
                raise RuntimeError("Installer-Module not found (core.updater).")

            if is_hub:
                self._log(
                    "Install requested for %s (hub self-install) → %s"
                    % (module_id, _current_hub_dir())
                )
            else:
                self._log(
                    "Install requested for %s → %s"
                    % (module_id, _compute_tool_dir_for(module_id))
                )

            try:
                msg = updater.download_and_install(upd, manifest_url=DEFAULT_MANIFEST)
            except TypeError:
                msg = updater.download_and_install(upd)

            if is_hub:
                _postfix_relocate_hub()

            # danach Registry und Catalog neu bauen
            discover_and_register()
            self.reload_all()

            # Shelf schreiben
            self.add_tool_to_shelf(module_id, self.catalog_cache.get(module_id, {}))
            self.persist_shelves("PjotsZenTools")

            self._log(f"Install done, shelf added for {module_id}")
            return True, (msg or "Installed.")
        except Exception as e:
            self._log(f"Install exception: {e}", level="ERROR")
            traceback.print_exc()
            return False, str(e)

    def safe_uninstall(self, module_id: str) -> Tuple[bool, str]:
        import shutil, time as _time
        self._log(f"_safe_uninstall start: {module_id}")
        res = None
        if updater:
            try:
                self._log("Try updater.uninstall(module_id=...)")
                res = updater.uninstall(module_id=module_id)
            except TypeError:
                try:
                    self._log("Updater signature fallback: uninstall(module_id)")
                    res = updater.uninstall(module_id)
                except Exception as e:
                    self._log(f"Updater uninstall (legacy) exception: {e}", level="WARN")
                    res = None
            except Exception as e:
                self._log(f"Updater uninstall exception: {e}", level="WARN")
                res = None
        if not res:
            reg = get_tools() or {}
            inst_meta = reg.get(module_id) or {}
            path = inst_meta.get("path") or (self.catalog_cache.get(module_id) or {}).get("path")
            self._log(f"Fallback rmtree path={path}")
            if path and os.path.exists(path):
                try:
                    shutil.rmtree(path)
                except Exception:
                    _time.sleep(0.1)
                    shutil.rmtree(path)

        try:
            self._log("discover_and_register()"); discover_and_register()
        finally:
            self._log("reload_all()"); self.reload_all()

        try:
            self._log(f"Remove shelf button for {module_id}")
            self.remove_tool_from_shelf(module_id)
            self.sync_shelf_with_registry()
            self.persist_shelves("PjotsZenTools")
        except Exception as e:
            self._log(f"Shelf cleanup/persist exception: {e}", level="WARN")

        # Registry Flag (installed=False) best effort
        try:
            import pjots_zen_tools.hub.registry as reg  # type: ignore
            if not reg.set_installed(module_id, False):
                reg.unregister_tool(module_id)
            self._log(f"Registry updated for {module_id}: installed=False")
        except Exception as e:
            self._log(f"Registry update failed for {module_id}: {e}", level="WARN")

        self._log(f"_safe_uninstall done: {module_id}")
        return True, "Tool uninstalled."

    # ---------- Shelf ----------
    def _ensure_shelf(self, name: str = "PjotsZenTools") -> Optional[str]:
        if not self.cmds: return None
        name = self._norm_shelf_name(name); cmds = self.cmds
        if not cmds.shelfTabLayout("ShelfLayout", q=True, ex=True):
            self._log("ShelfLayout not found (headless?)", level="WARN"); return None
        if not cmds.shelfLayout(name, q=True, ex=True):
            self._log(f"Create shelf '{name}'"); cmds.shelfLayout(name, p="ShelfLayout")
        return name

    @staticmethod
    def _norm_shelf_name(n: str) -> str:
        n = (n or "").strip()
        if n.lower().startswith("shelf_"): n = n[6:]
        if n.lower().endswith(".mel"): n = n[:-4]
        return n

    def add_tool_to_shelf(self, module_id: str, meta: dict) -> None:
        if not self.cmds:
            return
        shelf = self._ensure_shelf()
        if not shelf:
            return
        cmds = self.cmds
        title = meta.get("title", module_id)
        self._log(f"Add shelf button: {module_id} (label='{title}')")

        man = (meta or {}).get("manifest_entry") or {}
        fallback_entry = man.get("entry") or meta.get("entry")
        if not fallback_entry:
            self._log(f"No fallback_entry for {module_id}; skip", level="WARN")
            return
        try:
            mod, func = str(fallback_entry).split(":", 1)
        except ValueError:
            self._log(f"Invalid fallback_entry '{fallback_entry}' for {module_id}", level="WARN")
            return

        py_cmd = (
            "import importlib\n"
            "import pjots_zen_tools.hub.registry as _reg\n"
            f"_mid='{module_id}'\n"
            "_m=_reg.get_tools().get(_mid)\n"
            "_l=_m.get('launcher') if _m else None\n"
            "if callable(_l):\n"
            "    _l()\n"
            "else:\n"
            f"    _m = importlib.import_module('{mod}')\n"
            f"    getattr(_m, '{func}')()\n"
        )

        # Doppelte Buttons entfernen
        self.remove_tool_from_shelf(module_id)

        # Icon priorisieren: Registry → META/Manifest → Default
        try:
            reg_icon = (get_tools() or {}).get(module_id, {}).get("icon") or ""
        except Exception:
            reg_icon = ""
        icon = reg_icon or meta.get("icon") or man.get("icon") or ""
        if R:
            try:
                icon = R.full_icon_path(icon) or icon  # type: ignore
            except Exception:
                pass
        if not icon or not os.path.exists(icon):
            icon = "commandButton.png"  # Maya-Default

        try:
            cmds.shelfButton(
                p=shelf,
                i=icon,
                l=title,
                ann=f"zen:{module_id}",
                c=py_cmd,
                stp="python",
                imageOverlayLabel=title[:4]
            )
        except Exception:
            try:
                cmds.shelfButton(
                    p=shelf,
                    i=icon,
                    l=title,
                    ann=f"zen:{module_id}",
                    c=py_cmd,
                    stp="python"
                )
            except Exception as e:
                self._log(f"shelfButton create failed: {e}", level="ERROR")

    def remove_tool_from_shelf(self, module_id: str) -> None:
        if not self.cmds: return
        cmds = self.cmds; shelf = self._ensure_shelf()
        if not shelf: return
        try:
            if shelves_api and hasattr(shelves_api, "remove_module_button"):
                removed = shelves_api.remove_module_button(cmds, shelf, module_id)
                self._log(f"Removed {removed} shelf button(s) for {module_id}")
            else:
                children = cmds.shelfLayout(shelf, q=True, ca=True) or []
                removed = 0
                for ch in children:
                    try: ann = cmds.shelfButton(ch, q=True, ann=True)
                    except Exception: ann = None
                    if ann == f"zen:{module_id}":
                        try: cmds.deleteUI(ch); removed += 1
                        except Exception: pass
                self._log(f"Removed {removed} shelf button(s) for {module_id} (legacy)")
        except Exception as e:
            self._log(f"remove_tool_from_shelf error: {e}", level="WARN")

    def sync_shelf_with_registry(self, name: str = "PjotsZenTools") -> None:
        if not self.cmds or not shelves_api or not hasattr(shelves_api, "sync_shelf_with_registry"):
            return
        try:
            reg = get_tools() or {}
            removed = shelves_api.sync_shelf_with_registry(self.cmds, name, reg)
            self._log(f"Sync shelf removed {removed} stale button(s)")
        except Exception as e:
            self._log(f"sync_shelf_with_registry error: {e}", level="WARN")

    def persist_shelves(self, shelf_name: Optional[str] = None) -> None:
        if not self.cmds: return
        name = self._norm_shelf_name(shelf_name or "PjotsZenTools")
        try:
            if shelves_api and hasattr(shelves_api, "ensure_wrapper_file"):
                shelves_api.ensure_wrapper_file(
                    cmds=self.cmds, name=name, bootstrap=("hub", "vd"))
        except Exception as e:
            self._log(f"ensure_wrapper_file failed: {e}", level="WARN")
        try:
            import maya.mel as mel  # type: ignore
            prefs_dir = self.cmds.internalVar(userPrefDir=True)
            shelf_dir = os.path.join(prefs_dir, "shelves")
            os.makedirs(shelf_dir, exist_ok=True)
            path = os.path.join(shelf_dir, f"shelf_{name}").replace("\\", "/")
            self._log(f'Persist shelves: saveShelf "{name}" -> {path}')
            mel.eval(f'saveShelf "{name}" "{path}";')
        except Exception as e:
            self._log(f"saveShelf failed: {e}", level="ERROR")
