# hub_ui.py — Pjot's Zen Tools HUB (UI, delegiert an actions.HubActions)

from __future__ import annotations
import os, hashlib, time, json, traceback
from typing import Any, Callable, Dict, Optional


# Qt & Theming
try:
    from pjots_zen_tools.core import qt_compat as QC
    QtCore, QtWidgets, QtGui = QC.QtCore, QC.QtWidgets, QC.QtGui
except Exception:
    try:
        from PySide2 import QtCore, QtWidgets, QtGui  # type: ignore
    except Exception:
        from PySide6 import QtCore, QtWidgets, QtGui  # type: ignore
try:
    Signal = QtCore.Signal          # PySide2/PySide6
except AttributeError:
    from PyQt5.QtCore import pyqtSignal as Signal  # Fallback für PyQt

try:
    from pjots_zen_tools.core.theming import apply_styles
except Exception:
    def apply_styles(*_args, **_kwargs): pass

# Ressources
R = None
try:
    from pjots_zen_tools import resources as R
except Exception:
    R = None

def _res_path(p: str) -> str:
    """Robuste Pfadauflösung für Icons (QRC, resources, package-relative)."""
    if not p:
        return ""
    # 1) Qt Resource (:prefix) direkt durchlassen
    if p.startswith(":"):
        return p

    # 2) resources.full_icon_path (falls vorhanden)
    if R:
        try:
            rp = R.full_icon_path(p)
            if rp and os.path.exists(rp):
                return rp
        except Exception:
            pass

    # 3) Absoluter Pfad?
    if os.path.isabs(p) and os.path.exists(p):
        return p

    # 4) Relativ zum Modulordner (…/pjots_zen_tools/hub/)
    here = os.path.dirname(__file__)
    cand = os.path.join(here, p)
    if os.path.exists(cand):
        return cand

    # 5) Speziell: icons/ liegt neben dieser Datei
    cand = os.path.join(here, "icons", os.path.basename(p))
    if os.path.exists(cand):
        return cand

    # 6) Fallback: eine Ebene höher (…/pjots_zen_tools/), dann hub/icons
    pkg_root = os.path.dirname(here)
    cand = os.path.join(pkg_root, "hub", "icons", os.path.basename(p))
    if os.path.exists(cand):
        return cand

    # Not found → unverändert zurück (Qt probiert's trotzdem)
    return p

HUB_LOGO     = getattr(R, "HUB_LOGO", None) if R else None
HUB_LOGO_HD  = getattr(R, "HUB_LOGO_HD", None) if R else None

ICON_CLOUD   = _res_path(getattr(R, "ICON_CLOUD_SYNC", "icons/cloud_sync.png") if R else "icons/cloud_sync.png")
ICON_ACCOUNT = _res_path(getattr(R, "ICON_ACCOUNT",   "icons/account.png")     if R else "icons/account.png")

logo_path = _res_path(HUB_LOGO_HD or HUB_LOGO) if (HUB_LOGO_HD or HUB_LOGO) else ""

# Registry APIs (nur für Launch/Icons)
try:
    from pjots_zen_tools.hub.registry import get_tools
except Exception:
    def get_tools() -> Dict[str, Dict[str, Any]]: return {}

# Actions (Business-Logik)
from pjots_zen_tools.hub.actions import HubActions


TERMS_URL = "https://get.pjotszen.tools/terms.html"      # EN
PRIVACY_URL = "https://get.pjotszen.tools/privacy.html"  # EN

def _open_url(url: str):
    try:
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(url))
    except Exception:
        pass

class _CheckUpdatesWorker(QtCore.QObject):
    finished = Signal(int, str)  # (count, error_message)

    def __init__(self, actions, parent=None):
        super().__init__(parent)
        self._actions = actions

    @QtCore.Slot()
    def run(self):
        try:
            cnt = int(self._actions.force_check_updates())
            self.finished.emit(cnt, "")
        except Exception as e:
            self.finished.emit(-1, str(e))

# ---------------------------------------------------------------------
# UI-Only Helpers
# ---------------------------------------------------------------------
def _maya_main_window():
    try:
        from shiboken2 import wrapInstance
    except Exception:
        try:
            from shiboken6 import wrapInstance
        except Exception:
            wrapInstance = None
    if wrapInstance is None:
        return None
    try:
        from maya import OpenMayaUI as omui
        ptr = omui.MQtUtil.mainWindow()
        if not ptr:
            return None
        try:
            return wrapInstance(int(ptr), QtWidgets.QWidget)
        except Exception:
            return None
    except Exception:
        return None


def _user_prefs_dir():
    try:
        import maya.cmds as _cmds  # type: ignore
        return _cmds.internalVar(userPrefDir=True).replace("\\", "/")
    except Exception:
        return os.path.join(os.path.expanduser("~"), ".pjots_zen").replace("\\", "/")

_IMAGE_CACHE_DIR = os.path.join(_user_prefs_dir(), "image_cache")

def _ensure_image_cache_dir():
    os.makedirs(_IMAGE_CACHE_DIR, exist_ok=True)
    return _IMAGE_CACHE_DIR

def _download_bytes(url: str, timeout_s: int = 8) -> bytes:
    import urllib.request
    req = urllib.request.Request(url, headers={"User-Agent": "PjotsZenHub/1.0"})
    with urllib.request.urlopen(req, timeout=timeout_s) as r:
        return r.read()

def _cached_image_bytes(url: str, ttl_seconds: int = 86400) -> bytes | None:
    try:
        cdir = _ensure_image_cache_dir()
        h = hashlib.sha1(url.encode("utf-8")).hexdigest()
        ext = os.path.splitext(url.split("?", 1)[0])[1] or ".img"
        fpath = os.path.join(cdir, f"{h}{ext}")
        if os.path.exists(fpath) and (time.time() - os.path.getmtime(fpath)) < ttl_seconds:
            with open(fpath, "rb") as f:
                return f.read()
        data = _download_bytes(url)
        with open(fpath, "wb") as f:
            f.write(data)
        return data
    except Exception:
        return None

class IconButton32(QtWidgets.QToolButton):
    """32×32 Icon-Button mit sauberem Hover/Pressed-Feedback."""
    def __init__(self, icon: QtGui.QIcon | None, tooltip: str, fallback_text: str = "", parent=None):
        super().__init__(parent)
        self.setToolTip(tooltip)
        self.setAutoRaise(True)
        self.setCursor(QtCore.Qt.PointingHandCursor)
        self.setPopupMode(QtWidgets.QToolButton.InstantPopup)  # Menu (falls gesetzt) öffnet sofort
        self.setProperty("role", "icon")  # fürs QSS
        self.setIconSize(QtCore.QSize(32, 32))
        self.setFixedSize(44, 44)  # 32 Icon + Padding
        if icon:
            self.setIcon(icon)
        else:
            # Fallback, falls Icon fehlt
            self.setText(fallback_text or "•")

    @staticmethod
    def qicon_from_path(path: str | None) -> QtGui.QIcon | None:
        if not path: return None
        pix = QtGui.QPixmap(path)
        if pix.isNull(): return None
        ico = QtGui.QIcon()
        for mode in (QtGui.QIcon.Normal, QtGui.QIcon.Active, QtGui.QIcon.Selected, QtGui.QIcon.Disabled):
            ico.addPixmap(pix, mode, QtGui.QIcon.Off)
        return ico

class CenterNotice(QtWidgets.QDialog):
    """Rounded, shadowed, centered modal dialog for short success/info messages."""
    def __init__(self, parent=None, title: str = "Notice", message: str = "OK"):
        super().__init__(parent)
        self.setObjectName("ZenRoot")
        try:
            apply_styles(self)
        except Exception:
            pass
        self.setWindowFlags(QtCore.Qt.Dialog | QtCore.Qt.FramelessWindowHint)
        self.setModal(True)
        self.setAttribute(QtCore.Qt.WA_TranslucentBackground, True)

        outer = QtWidgets.QVBoxLayout(self)
        outer.setContentsMargins(0, 0, 0, 0)

        card = QtWidgets.QFrame(self)
        card.setObjectName("ZenPopoverContent")  # reuse same QSS styling
        card.setStyleSheet("QFrame#ZenPopoverContent { background:rgba(21,27,45,240); border-radius:20px; }")
        eff = QtWidgets.QGraphicsDropShadowEffect(card)
        eff.setBlurRadius(24); eff.setOffset(0, 6); eff.setColor(QtGui.QColor(0,0,0,160))
        card.setGraphicsEffect(eff)

        lay = QtWidgets.QVBoxLayout(card)
        lay.setContentsMargins(20, 20, 20, 16)
        lay.setSpacing(10)

        lbl_title = QtWidgets.QLabel(title)
        lbl_title.setStyleSheet("font-size:16px; font-weight:600;")
        lbl_msg = QtWidgets.QLabel(message)
        lbl_msg.setWordWrap(True)
        btn_ok = QtWidgets.QPushButton("OK")
        btn_ok.setFixedWidth(88)
        btn_ok.setProperty("variant", "primary")
        btn_ok.clicked.connect(self.accept)

        lay.addWidget(lbl_title)
        lay.addWidget(lbl_msg)
        row = QtWidgets.QHBoxLayout(); row.addStretch(1); row.addWidget(btn_ok)
        lay.addLayout(row)

        outer.addWidget(card)

        # sensible default size
        self.resize(420, card.sizeHint().height() + 20)

    def showEvent(self, e):
        super().showEvent(e)
        # center on parent window
        try:
            host = self.parent().window() if self.parent() else None
            geom = host.frameGeometry() if host else QtWidgets.QApplication.desktop().availableGeometry(self)
            x = geom.center().x() - self.width() // 2
            y = geom.center().y() - self.height() // 2
            self.move(x, y)
        except Exception:
            pass

class ArrowPopover(QtWidgets.QFrame):
    """Popover with shadow + arrow; stays within parent window and can center arrow under anchor."""
    closed = Signal()

    # geometry constants
    _MARGIN = 8          # outer margin to widget edge (all sides)
    _TOP_PAD = 14        # top padding (space where arrow attaches)
    _RADIUS = 20
    _ARROW_W = 16
    _ARROW_H = 14

    def __init__(self, parent=None):
        super().__init__(parent, QtCore.Qt.Popup | QtCore.Qt.FramelessWindowHint)
        self.setWindowFlag(QtCore.Qt.NoDropShadowWindowHint, True)
        self.setAttribute(QtCore.Qt.WA_TranslucentBackground, True)
        self.setObjectName("ZenPopover")
        self._arrow_x = None  # arrow tip x within inner rect (set in popupAt)

        self._content = QtWidgets.QFrame(self)
        self._content.setObjectName("ZenPopoverContent")
        self._content.setLayout(QtWidgets.QVBoxLayout())
        self._content.layout().setContentsMargins(16, 16, 16, 16)
        self._content.layout().setSpacing(12)

        eff = QtWidgets.QGraphicsDropShadowEffect(self)
        eff.setBlurRadius(24); eff.setOffset(0, 4); eff.setColor(QtGui.QColor(0,0,0,160))
        self._content.setGraphicsEffect(eff)

        QtWidgets.QApplication.instance().installEventFilter(self)

    def setContentWidget(self, w: QtWidgets.QWidget):
        lay = self._content.layout()
        while lay.count():
            it = lay.takeAt(0)
            if it.widget(): it.widget().deleteLater()
        lay.addWidget(w)

    def popupAt(self, anchor: QtWidgets.QWidget, dx: int = 0, gap: int = 0, clamp_to_parent: bool = True):
        """
        Show popover so arrow sits (as much as possible) centered under the anchor.
        'gap' is the visible space between anchor bottom and arrow tip.
        Clamps inside parent window by default.
        """
        # anchor center (global)
        anc_rect = anchor.rect()
        anc_center_global = anchor.mapToGlobal(anc_rect.center())

        # target area to clamp
        if clamp_to_parent and self.parent():
            win = self.parent().window()
            top_left_global = win.mapToGlobal(QtCore.QPoint(0, 0))
            win_geo = QtCore.QRect(top_left_global, win.size())
        else:
            win_geo = QtWidgets.QApplication.primaryScreen().availableGeometry()

        # preliminary left: center popover horizontally on anchor icon
        left = anc_center_global.x() - self.width() // 2
        top  = anchor.mapToGlobal(QtCore.QPoint(0, anc_rect.height())).y()

        # arrow tip offset from widget top (we draw inner rect starting at _TOP_PAD)
        ARROW_TIP_OFFSET = self._TOP_PAD

        # apply requested x shift
        left += dx

        # clamp inside window
        left = max(win_geo.left() + self._MARGIN,
                   min(left, win_geo.right() - self.width() - self._MARGIN))
        # y: align just under anchor, subtract arrow offset so arrow tip is near anchor
        y = top + gap - ARROW_TIP_OFFSET
        y = max(win_geo.top() + self._MARGIN,
                min(y, win_geo.bottom() - self.height() - self._MARGIN))

        # compute arrow_x within inner rect (local coords)
        inner_left = left + self._MARGIN
        # arrow local x = anchor_center_x - (popover_left + inner_margin)
        arrow_x = anc_center_global.x() - inner_left
        # keep arrow inside rounded corners
        min_ax = self._RADIUS + self._ARROW_W // 2
        max_ax = (self.width() - 2 * self._MARGIN) - (self._RADIUS + self._ARROW_W // 2)
        self._arrow_x = max(min_ax, min(arrow_x, max_ax))

        self.move(left, y)
        self.show()

    def resizeEvent(self, ev):
        super().resizeEvent(ev)
        self._content.setGeometry(self._MARGIN, self._TOP_PAD,
                                  self.width() - 2 * self._MARGIN,
                                  self.height() - (self._TOP_PAD + self._MARGIN))

    def paintEvent(self, ev):
        p = QtGui.QPainter(self)
        p.setRenderHint(QtGui.QPainter.Antialiasing)
        p.setPen(QtCore.Qt.NoPen)
        p.setBrush(QtGui.QColor(21, 27, 45, 240))

        # inner rect
        r = QtCore.QRect(self._MARGIN, self._TOP_PAD,
                         self.width() - 2 * self._MARGIN,
                         self.height() - (self._TOP_PAD + self._MARGIN))
        p.drawRoundedRect(r, self._RADIUS, self._RADIUS)

        # arrow under account icon (centered), clamped in popupAt
        ax = int(self._arrow_x if self._arrow_x is not None else r.width() -16)
        tip = QtCore.QPoint(r.left() + ax, r.top() - (self._ARROW_H - 1))
        left = QtCore.QPoint(tip.x() - self._ARROW_W // 2, r.top())
        right = QtCore.QPoint(tip.x() + self._ARROW_W // 2, r.top())
        p.drawPolygon(QtGui.QPolygon([left, tip, right]))

    def eventFilter(self, obj, event):
        if event.type() == QtCore.QEvent.MouseButtonPress:
            if not self.geometry().contains(QtGui.QCursor.pos()):
                self.close()
        return super().eventFilter(obj, event)

    def closeEvent(self, ev):
        self.closed.emit()
        super().closeEvent(ev)

# ---------------------------------------------------------------------
# UI Widgets
# ---------------------------------------------------------------------
class RichTextView(QtWidgets.QTextBrowser):
    def __init__(self, parent=None, allow_remote=True, whitelist_hosts=None, cache_ttl=86400, max_px=1024):
        super().__init__(parent)
        self._allow_remote = bool(allow_remote)
        self._whitelist = set(whitelist_hosts or [])
        self._cache_ttl = int(cache_ttl)
        self._max_px = int(max_px)
        self.setOpenExternalLinks(True)
        self.setReadOnly(True)

    def loadResource(self, res_type, url):
        if res_type == QtGui.QTextDocument.ImageResource:
            u = QtCore.QUrl(url) if not isinstance(url, QtCore.QUrl) else url
            scheme = (u.scheme() or "").lower()
            if scheme == "data":
                return super().loadResource(res_type, u)
            if scheme in ("http", "https"):
                if not self._allow_remote:
                    return None
                if self._whitelist and (u.host() not in self._whitelist):
                    return None
                try:
                    b = _cached_image_bytes(u.toString(), ttl_seconds=self._cache_ttl)
                    if not b: return None
                    img = QtGui.QImage()
                    if not img.loadFromData(b): return None
                    if max(img.width(), img.height()) > self._max_px:
                        img = img.scaled(self._max_px, self._max_px, QtCore.Qt.KeepAspectRatio, QtCore.Qt.SmoothTransformation)
                    return img
                except Exception:
                    return None
            return super().loadResource(res_type, u)
        return super().loadResource(res_type, url)

class ScaledImage(QtWidgets.QLabel):
    def __init__(self, path: Optional[str], max_width: int = 260, parent=None):
        super().__init__(parent)
        self._orig = None
        self._max_w = max_width
        self.setAlignment(QtCore.Qt.AlignCenter)
        self.setStyleSheet("background:none; border:none;")
        if path and os.path.exists(path):
            pm = QtGui.QPixmap(path)
            if not pm.isNull():
                self._orig = pm
                self._rescale()
    def resizeEvent(self, e):
        super().resizeEvent(e); self._rescale()
    def _rescale(self):
        if not self._orig or self.width() <= 0: return
        self.setPixmap(self._orig.scaledToWidth(min(self.width(), self._max_w), QtCore.Qt.SmoothTransformation))

class IconToolButton(QtWidgets.QToolButton):
    def __init__(self, tooltip: str, emoji_fallback: str = "⋯", parent=None):
        super().__init__(parent)
        self.setToolTip(tooltip)
        self.setText(emoji_fallback)
        self.setAutoRaise(False)
        self.setCursor(QtCore.Qt.PointingHandCursor)
        self.setPopupMode(QtWidgets.QToolButton.InstantPopup)  # menu shows instantly
        self.setIconSize(QtCore.QSize(18, 18))
        self.setStyleSheet("QToolButton { padding: 6px; }")

class SectionLabel(QtWidgets.QLabel):
    def __init__(self, text: str, parent=None):
        super().__init__(text, parent)
        self.setAlignment(QtCore.Qt.AlignCenter)
        self.setProperty("role", "chip")

class LoginDialog(QtWidgets.QDialog):
    def __init__(self, parent=None, name:str="", email:str="", studio:bool=False, marketing:bool=False):
        super().__init__(parent)
        self.setObjectName("ZenRoot")
        try:
            apply_styles(self)
        except Exception:
            pass
        self.setWindowTitle("Login / Registration")
        self.setModal(True)
        self.setMinimumWidth(420)

        v = QtWidgets.QVBoxLayout(self)
        v.setContentsMargins(12,12,12,12); v.setSpacing(8)

        form = QtWidgets.QFormLayout()
        form.setLabelAlignment(QtCore.Qt.AlignRight)
        self.ed_name  = QtWidgets.QLineEdit(self);  self.ed_name.setPlaceholderText("John Doe")
        self.ed_email = QtWidgets.QLineEdit(self);  self.ed_email.setPlaceholderText("name@example.com")
        if name:  self.ed_name.setText(name)
        if email: self.ed_email.setText(email)
        form.addRow("Name:", self.ed_name)
        form.addRow("Email address:", self.ed_email)

        self.chk_studio    = QtWidgets.QCheckBox("Commercial license (Studios, client work)")
        self.chk_marketing = QtWidgets.QCheckBox("Send me emails on updates or new tools")
        self.chk_marketing.setChecked(False)  # nicht vorangekreuzt (DSGVO)
        self.chk_studio.setChecked(bool(studio))
        if bool(marketing):
            self.chk_marketing.setChecked(True)

        v.addLayout(form)
        v.addWidget(self.chk_studio)
        v.addWidget(self.chk_marketing)

        hint = QtWidgets.QLabel("New here? Just enter a valid email. You will recieve a confirmation email in order to get access to the tools.")
        hint.setWordWrap(True); hint.setProperty("role", "muted")
        v.addWidget(hint)

        btns = QtWidgets.QDialogButtonBox(QtWidgets.QDialogButtonBox.Ok | QtWidgets.QDialogButtonBox.Cancel)
        btns.accepted.connect(self._accept)
        btns.rejected.connect(self.reject)
        v.addWidget(btns)

    def _accept(self):
        em = (self.ed_email.text() or "").strip()
        if not em or ("@" not in em) or (len(em) < 6):
            QtWidgets.QMessageBox.warning(self, "Login", "Please enter a valid email address.")
            return
        self.accept()

    def data(self) -> dict:
        return {
            "name": (self.ed_name.text() or "").strip(),
            "email": (self.ed_email.text() or "").strip().lower(),
            "studio": bool(self.chk_studio.isChecked()),
            "marketing": bool(self.chk_marketing.isChecked()),
        }

    @staticmethod
    def get(parent=None, name:str="", email:str="", studio:bool=False, marketing:bool=False):
        dlg = LoginDialog(parent=parent, name=name, email=email, studio=studio, marketing=marketing)
        ok = dlg.exec_() == QtWidgets.QDialog.Accepted
        return ok, (dlg.data() if ok else {})


# ---------------------------------------------------------------------
# Hub UI
# ---------------------------------------------------------------------
class ZenToolsHub(QtWidgets.QWidget):
    def __init__(self, parent=None):
        super().__init__(parent)
        mw = _maya_main_window()
        if mw is not None:
            self.setParent(mw, QtCore.Qt.Window)
            self.setWindowModality(QtCore.Qt.NonModal)
            self.setWindowFlag(QtCore.Qt.WindowMinimizeButtonHint, True)
            self.setWindowFlag(QtCore.Qt.WindowCloseButtonHint, True)
            self.setWindowFlag(QtCore.Qt.WindowMaximizeButtonHint, False)
            self.setWindowTitle("Pjot’s ZenTools Hub")
        self.setObjectName("ZenRoot")
        apply_styles(self)
        self.setMinimumSize(860, 700)

        # Actions/State
        self._updates_index: Dict[str, Dict[str, Any]] = {}
        self._catalog_cache: Dict[str, Dict[str, Any]] = {}
        try:
            import maya.cmds as cmds  # type: ignore
            self._cmds = cmds
        except Exception:
            self._cmds = None

        # Actions-Backend mit Logger
        self.actions = HubActions(log=self._log_passthrough)

        # ----------------------------
        # Layout: Header + Content
        # ----------------------------
        root = QtWidgets.QVBoxLayout(self)
        root.setContentsMargins(0, 0, 0, 0)
        root.setSpacing(0)

        # Header (Search + Icons)
        self._build_header(root)
        self._account_popover = None

        # Content area (left list / right details)
        content = QtWidgets.QHBoxLayout()
        content.setContentsMargins(10, 0, 10, 10)
        content.setSpacing(8)
        root.addLayout(content, 1)

        lw = QtWidgets.QWidget(self)
        rw = QtWidgets.QWidget(self)
        lw.setFixedWidth(250)
        content.addWidget(lw)
        content.addWidget(rw, 1)

        left = QtWidgets.QVBoxLayout(lw)
        right = QtWidgets.QVBoxLayout(rw)
        for lay in (left, right):
            lay.setContentsMargins(0, 10, 0, 0)
            lay.setSpacing(8)

        # Left column
        if logo_path:
            left.addWidget(ScaledImage(logo_path, max_width=220))

        self.list = QtWidgets.QListWidget(self)
        self.list.setSelectionMode(QtWidgets.QAbstractItemView.SingleSelection)
        left.addWidget(self.list, 1)

        # Right column
        self.details = RichTextView(parent=self, allow_remote=True, whitelist_hosts=[], cache_ttl=86400, max_px=1024)
        self.details.setOpenExternalLinks(True)
        self.details.setFrameShape(QtWidgets.QFrame.NoFrame)
        right.addWidget(self.details, 1)
        self._apply_details_colors()

        self.btn_primary = QtWidgets.QPushButton("Install selected")
        self.btn_uninstall = QtWidgets.QPushButton("Uninstall selected")
        row = QtWidgets.QHBoxLayout()
        row.addWidget(self.btn_primary, 1)
        row.addWidget(self.btn_uninstall, 0)
        right.addLayout(row)

        # Debug area
        self._debug_wrap = QtWidgets.QWidget(self)
        dbg = QtWidgets.QVBoxLayout(self._debug_wrap)
        dbg.setContentsMargins(0, 0, 0, 0)
        dbg.setSpacing(4)
        hdr = QtWidgets.QHBoxLayout()
        dbg.addLayout(hdr)
        self._debug_toggle = QtWidgets.QCheckBox("Show Debug log")
        self._debug_copy = QtWidgets.QPushButton("Copy log")
        hdr.addWidget(self._debug_toggle)
        hdr.addStretch(1)
        hdr.addWidget(self._debug_copy)
        self._debug = QtWidgets.QPlainTextEdit(self)
        self._debug.setReadOnly(True)
        self._debug.setMaximumBlockCount(2000)
        self._debug.setStyleSheet("font-family: Consolas, Menlo, monospace; font-size:11px;")
        dbg.addWidget(self._debug, 1)
        self._debug_wrap.setVisible(False)
        right.addWidget(self._debug_wrap)

        # Signals
        # (self.search is already connected in _build_header)
        self.list.currentItemChanged.connect(self._update_details)
        self.list.itemSelectionChanged.connect(self._recalc_buttons)
        self.btn_primary.clicked.connect(self._on_primary_clicked)
        self.btn_uninstall.clicked.connect(self._uninstall_selected)
        self._debug_toggle.toggled.connect(self._debug_wrap.setVisible)
        self._debug_copy.clicked.connect(self._copy_debug_log)

        # Final UI state
        self._auto_check_updates_on_open(initial=True)
        self._recalc_buttons()
        self._refresh_account_button()
        self._log("Hub UI initialized")

    def _show_center_notice(self, title: str, message: str):
        dlg = CenterNotice(parent=self, title=title, message=message)
        dlg.exec_()

    def _build_header(self, parent_layout: QtWidgets.QVBoxLayout):
        bar = QtWidgets.QHBoxLayout()
        bar.setContentsMargins(10, 10, 6, 0)
        bar.setSpacing(6)

        # Search
        self.search = QtWidgets.QLineEdit(self)
        self.search.setPlaceholderText("Search tools… (title, description, tags)")
        self.search.textChanged.connect(self._filter)
        bar.addWidget(self.search, 1)

        # Check for updates (Cloud-Icon)
        cloud_icon = IconButton32.qicon_from_path(ICON_CLOUD)
        self.btn_check_icon = IconButton32(cloud_icon, "Check for updates", fallback_text="☁")
        self.btn_check_icon.clicked.connect(self._toggle_updates_popover)
        bar.addWidget(self.btn_check_icon, 0)

        # Account (Person-Icon mit Dropdown)
        acc_icon = IconButton32.qicon_from_path(ICON_ACCOUNT)
        self.btn_account = IconButton32(acc_icon, "Account", fallback_text="👤")
        self.btn_account.clicked.connect(self._toggle_account_popover)
        bar.addWidget(self.btn_account, 0)

        parent_layout.addLayout(bar)

    def _toggle_account_popover(self):
        pop = getattr(self, "_account_popover", None)
        if pop and pop.isVisible():
            pop.close()
            return
        self._show_account_popover()

    def _show_account_popover(self):
        self._account_popover = ArrowPopover(self)
        content = self._build_account_view() if self._profile_exists() else self._build_register_view()
        self._account_popover.setContentWidget(content)
        # Größe an Inhalt anpassen
        h = 260 if self._profile_exists() else 300
        self._account_popover.resize(360, h)
        self._account_popover.popupAt(self.btn_account, dx = -100, clamp_to_parent = False)
        self._account_popover.closed.connect(lambda: setattr(self, "_account_popover", None))

    def _build_account_view(self) -> QtWidgets.QWidget:
        prof = self._get_profile() or {}
        w = QtWidgets.QWidget()
        lay = QtWidgets.QVBoxLayout(w); lay.setContentsMargins(0,0,0,0); lay.setSpacing(12)

        # Header (Icon + Name + Mail)
        head = QtWidgets.QHBoxLayout(); head.setSpacing(12)
        icon_lbl = QtWidgets.QLabel()
        if ICON_ACCOUNT:
            pm = QtGui.QPixmap(ICON_ACCOUNT)
        else:
            pm = self.style().standardIcon(QtWidgets.QStyle.SP_ComputerIcon).pixmap(56,56)
        if isinstance(pm, QtGui.QPixmap):
            icon_lbl.setPixmap(pm.scaled(56,56, QtCore.Qt.KeepAspectRatio, QtCore.Qt.SmoothTransformation))
        else:
            icon_lbl.setPixmap(pm)
        icon_lbl.setMinimumSize(56,56)
        head.addWidget(icon_lbl, 0)

        col = QtWidgets.QVBoxLayout(); col.setSpacing(2)
        name = QtWidgets.QLabel(prof.get("name") or "Account")
        name.setStyleSheet("font-size:16px; font-weight:600;")
        mail = QtWidgets.QLabel((prof.get("email") or "").lower())
        mail.setStyleSheet("color:#A9B1C3;")
        col.addWidget(name); col.addWidget(mail)
        head.addLayout(col, 1)
        lay.addLayout(head)

        # Account popover content
        grid = QtWidgets.QGridLayout()
        grid.setHorizontalSpacing(24)
        grid.setVerticalSpacing(6)

        def muted(txt: str) -> QtWidgets.QLabel:
            lab = QtWidgets.QLabel(txt)
            lab.setStyleSheet("color:#A9B1C3;")
            return lab

        line = QtWidgets.QFrame()
        line.setFrameShape(QtWidgets.QFrame.HLine)
        line.setStyleSheet("color:rgba(255,255,255,0.08);")
        lay.addWidget(line)

        # Row: Commercial license
        grid.addWidget(muted("Commercial license:"), 0, 0)

        remote_active = bool(getattr(self.actions, "_commercial_active", False))
        if remote_active:
            grid.addWidget(QtWidgets.QLabel("Yes"), 0, 1)
        else:
            # "No" + CTA button → opens get_commercial.php with current email
            row = QtWidgets.QWidget()
            hl = QtWidgets.QHBoxLayout(row)
            hl.setContentsMargins(0, 0, 0, 0)
            hl.setSpacing(8)
            hl.addWidget(QtWidgets.QLabel("No"))
            btn_get = QtWidgets.QPushButton("Get commercial license")
            # Optional Zen variant:
            # btn_get.setProperty("variant", "primary")
            btn_get.clicked.connect(self._open_get_commercial)
            hl.addWidget(btn_get)
            hl.addStretch(1)
            grid.addWidget(row, 0, 1)

        # Row: Marketing opt-in
        grid.addWidget(muted("Marketing opt-in:"), 1, 0)
        grid.addWidget(QtWidgets.QLabel("Yes" if prof.get("marketing") else "No"), 1, 1)

        lay.addLayout(grid)

        # --- Inline Legal als normaler Text mit Links ---
        legal = QtWidgets.QLabel(
            'See <a href="terms">Terms of Use</a> and '
            '<a href="privacy">Privacy Policy</a>.'
        )
        legal.setTextFormat(QtCore.Qt.RichText)
        legal.setOpenExternalLinks(False)  # wir fangen die Links selber ab
        legal.linkActivated.connect(lambda href: _open_url(TERMS_URL) if href == "terms" else _open_url(PRIVACY_URL))
        legal.setStyleSheet("color:#A9B1C3;")  # muted
        lay.addWidget(legal)

        # --- Logout ---
        line = QtWidgets.QFrame()
        line.setFrameShape(QtWidgets.QFrame.HLine)
        line.setStyleSheet("color:rgba(255,255,255,0.08);")
        lay.addWidget(line)

        btn_out = QtWidgets.QPushButton("Logout")
        btn_out.setCursor(QtCore.Qt.PointingHandCursor)
        btn_out.setStyleSheet("QPushButton{color:#FF6464; font-weight:600; text-align:left;} QPushButton:hover{opacity:0.9;}")
        btn_out.clicked.connect(self._menu_logout)
        lay.addWidget(btn_out)

        return w

    def _open_get_commercial(self) -> None:
        url = self.actions.get_commercial_url()
        QtGui.QDesktopServices.openUrl(QtCore.QUrl(url))

    def _register_from_popover(self):
        name = self._reg_name.text().strip()
        email = self._reg_email.text().strip().lower()
        studio = bool(self._reg_studio.isChecked())
        marketing = bool(self._reg_marketing.isChecked())
        ok, _msg = self.actions.register_email(email, name=name, studio=studio, marketing=marketing)

        # decide wording based on entitlements containing the email
        def _contains_email(obj, em) -> bool:
            if not em: return False
            if isinstance(obj, str): return obj.strip().lower() == em
            if isinstance(obj, dict): return any(_contains_email(k, em) or _contains_email(v, em) for k, v in obj.items())
            if isinstance(obj, (list, tuple, set)): return any(_contains_email(x, em) for x in obj)
            return False

        if ok:
            ent = self.actions.entitlements or {}
            logged_in = _contains_email(ent, email)
            if logged_in:
                self._show_center_notice("Register", "Successfully logged in.")
            else:
                self._show_center_notice("Register", f"Successfully registered — a confirmation email was sent to {email}.")
            self._reload_catalog(); self._populate(); self._refresh_account_button()
            if getattr(self, "_account_popover", None):
                self._account_popover.setContentWidget(self._build_account_view())
        else:
            QtWidgets.QMessageBox.warning(self, "Register", "Registration failed. Please try again.")


    def _build_register_view(self) -> QtWidgets.QWidget:
        w = QtWidgets.QWidget()
        lay = QtWidgets.QVBoxLayout(w); lay.setContentsMargins(0,0,0,0); lay.setSpacing(10)

        title = QtWidgets.QLabel("Create your account")
        title.setStyleSheet("font-size:16px; font-weight:600;")
        lay.addWidget(title)

        self._reg_name  = QtWidgets.QLineEdit(); self._reg_name.setPlaceholderText("Full name")
        self._reg_email = QtWidgets.QLineEdit(); self._reg_email.setPlaceholderText("Email")
        self._reg_studio = QtWidgets.QCheckBox("Commercial license")
        self._reg_marketing = QtWidgets.QCheckBox("Marketing opt-in")
        for e in (self._reg_name, self._reg_email): e.setMinimumWidth(260)
        lay.addWidget(self._reg_name); lay.addWidget(self._reg_email)
        box = QtWidgets.QVBoxLayout(); box.setSpacing(6)
        box.addWidget(self._reg_studio); box.addWidget(self._reg_marketing)
        lay.addLayout(box)

        row = QtWidgets.QHBoxLayout(); row.addStretch(1)
        btn = QtWidgets.QPushButton("Register")
        btn.clicked.connect(self._register_from_popover)
        row.addWidget(btn)
        lay.addLayout(row)

        hint = QtWidgets.QLabel('<a href="terms">Terms of Use</a> · <a href="privacy">Privacy Policy</a>')
        hint.setTextFormat(QtCore.Qt.RichText); hint.setOpenExternalLinks(False)
        hint.linkActivated.connect(lambda href: _open_url(TERMS_URL) if href=="terms" else _open_url(PRIVACY_URL))
        hint.setStyleSheet("color:#A9B1C3;")
        lay.addWidget(hint)

        return w

    def _rebuild_account_menu(self):
        """Build or rebuild the account dropdown menu."""
        m = QtWidgets.QMenu(self)

        if not self._profile_exists():
            # Not registered: one action to trigger the login/register dialog
            act_login = m.addAction("Login / Register…")
            act_login.triggered.connect(self._register_email)
            m.addSeparator()
            act_terms = m.addAction("Terms of Use (EN)")
            act_terms.triggered.connect(lambda: _open_url(TERMS_URL))
            act_priv = m.addAction("Privacy Policy (EN)")
            act_priv.triggered.connect(lambda: _open_url(PRIVACY_URL))
        else:
            prof = self._get_profile()
            # Show a tiny read-only summary (disabled actions for visual grouping)
            title = m.addAction(f"{prof.get('name') or 'Account'}")
            title.setEnabled(False)
            email = m.addAction(prof.get("email", "").lower())
            email.setEnabled(False)

            # Status lines
            m.addSeparator()
            lic = "Yes" if prof.get("studio") else "No"
            mark = "Yes" if prof.get("marketing") else "No"
            a1 = m.addAction(f"Commercial license: {lic}"); a1.setEnabled(False)
            a2 = m.addAction(f"Marketing opt-in: {mark}");  a2.setEnabled(False)

            m.addSeparator()
            act_terms = m.addAction("Terms of Use (EN)")
            act_terms.triggered.connect(lambda: _open_url(TERMS_URL))
            act_priv = m.addAction("Privacy Policy (EN)")
            act_priv.triggered.connect(lambda: _open_url(PRIVACY_URL))

            m.addSeparator()
            act_logout = m.addAction("Logout")
            act_logout.triggered.connect(self._menu_logout)

        self.btn_account.setMenu(m)

    def _menu_logout(self):
        self._perform_logout()
        self._refresh_account_button()
        self._reload_catalog(); self._populate()
        if getattr(self, "_account_popover", None):
            self._account_popover.close()

    def _on_primary_clicked(self) -> None:
        it = self.list.currentItem()
        if not it:
            return
        mode = getattr(self, "_primary_mode", None)
        if mode == "install":
            self._install_selected()
        elif mode == "update":
            self._update_selected()
        elif mode == "launch":
            self._launch_selected()
        elif mode == "buy":
            mid = it.data(QtCore.Qt.UserRole)
            meta = self._catalog_cache.get(mid, {}) or {}
            man = meta.get("manifest_entry") or {}
            buy_url = man.get("buy_url") or "https://get.pjotszen.tools/"
            QtGui.QDesktopServices.openUrl(QtCore.QUrl(buy_url))
        else:
            pass

    def _apply_details_colors(self):
        pal = self.details.palette()
        text    = pal.color(QtGui.QPalette.Text).name()
        link    = pal.color(QtGui.QPalette.Link).name()
        visited = pal.color(QtGui.QPalette.LinkVisited).name()

        css = (
            # Grundtypografie
            "body {"
            f"  color:{text}; background:transparent; font-size:13px; line-height:1.45;"
            "  -qt-block-indent:0; -qt-paragraph-type:empty; }"
            "h1, h2, h3 { margin:0 0 6px 0; font-weight:600; }"
            "h2 { font-size:18px; }"
            "p  { margin:6px 0; }"
            "ul,ol { margin:6px 0 6px 18px; }"
            ".muted { color:#A9B1C3; opacity:.95; }"
            # Header-Table (Logo links, Content rechts)
            ".head { width:100%; border-collapse:collapse; margin:0 0 8px 0; }"
            ".head td { vertical-align:top; }"
            ".logo { width:64px; padding:0 10px 0 0; }"
            ".logo img { display:block; height:128px; border-radius:8px; }"
            ".head .meta { margin:0 0 4px 0; }"
            # Badges (Chips)
            ".badges { border-collapse:separate; border-spacing:12px 8px; margin:6px 0 0 0; }"
            ".badges td { padding:0; }"
            ".badge {"
            "  display:inline-block;"
            "  padding:4px 12px;"
            "  line-height:1.1;"                         # kompakter Textblock
            "  border:1px solid rgba(255,255,255,0.08);"  # <- wichtig: irgendeine Border setzen
            "  border-radius:14px;"                       # runde Ecken
            "  background-clip:padding;"                  # wird von Qt meist toleriert
            "  white-space:nowrap; font-size:11px; font-weight:600;"
            "}"
            ".badge-ok   { color:#78D28B; }"
            ".badge-no   { color:#FFAAAA; }"
            ".badge-warn { color:#FFAD5C; }"
            ".badge-info { color:#6EA8FE; }"
            # Links
            f"a {{ color:{link}; text-decoration:none; }}"
            "a:hover { text-decoration:underline; }"
            f"a:visited {{ color:{visited}; }}"
        )
        self.details.document().setDefaultStyleSheet(css)

    def changeEvent(self, ev):
        super().changeEvent(ev)
        if ev.type() == QtCore.QEvent.PaletteChange:
            self._apply_details_colors()



    # ---------- Logging ----------
    def _log(self, msg: str, level: str = "INFO") -> None:
        line = f"[ZenToolsHub][{level}] {msg}"
        print(line)
        try: self._debug.appendPlainText(line)
        except Exception: pass

    def _log_passthrough(self, msg: str, level: str = "INFO"):
        self._log(msg, level)

    def _copy_debug_log(self) -> None:
        try:
            QtWidgets.QApplication.clipboard().setText(self._debug.toPlainText())
            QtWidgets.QMessageBox.information(self, "Debug", "Log copied to clipboard.")
        except Exception:
            pass

    # ---------- Styling helpers ----------
    @staticmethod
    def _repolish(*widgets):
        for w in widgets:
            try:
                w.style().unpolish(w); w.style().polish(w); w.update()
            except Exception:
                pass

    def _set_variant(self, btn: QtWidgets.QPushButton, name: Optional[str]):
        try:
            btn.setProperty("variant", name or "")
            self._repolish(btn)
        except Exception:
            pass

    def _set_destructive(self, btn: QtWidgets.QPushButton, on: bool):
        try:
            btn.setProperty("destructive", bool(on))
            self._repolish(btn)
        except Exception:
            pass

    # ---------- Data bridge ----------
    def _reload_catalog(self) -> None:
        self._catalog_cache = dict(self.actions.catalog_cache)
        self._updates_index = dict(self.actions.updates_index)
        self._email = self.actions.email
        self._entitlements = dict(self.actions.entitlements)
        self._allowed_ids = set(self.actions.allowed_ids)

    def _current_mid(self) -> Optional[str]:
        it = self.list.currentItem()
        return it.data(QtCore.Qt.UserRole) if it else None

    def _populate(self, prefer_mid: Optional[str] = None) -> None:
        self.list.clear()
        def _title_key(kv): return kv[1].get("title", kv[0]).lower()

        for mid, meta in sorted(self._catalog_cache.items(), key=_title_key):
            title = meta.get("title", mid)
            lv = meta.get("version", "0.0.0")
            rv = meta.get("remote_version")
            installed = bool(meta.get("installed"))

            if installed:
                label = f"{title}  •  {lv}"
            else:
                # nicht installiert
                label = f"{title}" + (f"  ")

            if not installed:
                label += "   ⬇"
            elif rv and self._semver_lt(lv, rv):
                label += "   ⬆"

            item = QtWidgets.QListWidgetItem(label)

            icon_path = meta.get("icon")
            if icon_path and R:
                try:
                    icon_path = R.full_icon_path(icon_path) or icon_path
                except Exception:
                    pass
            if icon_path:
                item.setIcon(QtGui.QIcon(icon_path))

            item.setData(QtCore.Qt.UserRole, mid)
            item.setData(QtCore.Qt.UserRole + 1,
                         (meta.get("title", ""), meta.get("description", ""), " ".join(meta.get("tags", [])).lower()))
            self.list.addItem(item)

        target_mid = prefer_mid or getattr(self, "_last_selected_mid", None)
        target_row = None
        if target_mid:
            for i in range(self.list.count()):
                if self.list.item(i).data(QtCore.Qt.UserRole) == target_mid:
                    target_row = i
                    break

        if self.list.count():
            self.list.setCurrentRow(target_row if target_row is not None else 0)

        self._update_details(self.list.currentItem())
        self._recalc_buttons()
        self._log("List populated")


    @staticmethod
    def _semver_lt(a: str, b: str) -> bool:
        def norm(v):
            v = (v or "0").replace("-", ".")
            return [int(x) if x.isdigit() else x for x in v.split(".")]
        na, nb = norm(a), norm(b)
        for xa, xb in zip(na, nb):
            if xa == xb: continue
            return xa < xb
        return len(na) < len(nb)

    def _rescan_tools(self) -> None:
        cur = self._current_mid()
        self.actions.rescan()
        self._reload_catalog()
        self._populate(prefer_mid=cur)

    # ---------- List/Details ----------
    def _filter(self, text: str) -> None:
        t = (text or "").lower().strip()
        for i in range(self.list.count()):
            it = self.list.item(i)
            if not t:
                it.setHidden(False); continue
            title, desc, tags = it.data(QtCore.Qt.UserRole + 1) or ("","","")
            hay = " ".join([it.text().lower(), title.lower(), desc.lower(), tags.lower()])
            it.setHidden(t not in hay)

    def _update_details(self, item: Optional[QtWidgets.QListWidgetItem]) -> None:
        import os
        if not item:
            self.details.setHtml("<i>No tool selected.</i>")
            return

        mid  = item.data(QtCore.Qt.UserRole)
        meta = self._catalog_cache.get(mid, {}) or {}
        man  = meta.get("manifest_entry") or {}
        rv   = meta.get("remote_version")
        lv   = meta.get("version", "0.0.0")
        desc = meta.get("description", "")
        tags = ", ".join(meta.get("tags", [])) or "-"

        installed   = bool(meta.get("installed"))
        allowed_ids = getattr(self, "_allowed_ids", set()) or set()
        entitled    = (mid in allowed_ids)

        # --- Email/Entitlements für Hinweise
        def _contains_email(obj, email: str) -> bool:
            if not email: return False
            if isinstance(obj, str): return obj.strip().lower() == email
            if isinstance(obj, dict): return any(_contains_email(k, email) or _contains_email(v, email) for k, v in obj.items())
            if isinstance(obj, (list, tuple, set)): return any(_contains_email(x, email) for x in obj)
            return False
        email       = (getattr(self, "_email", "") or "").strip().lower()
        ent_data    = getattr(self, "_entitlements", {}) or {}
        email_known = _contains_email(ent_data, email)

        # --- Icon (links)
        icon_path = meta.get("icon")
        if icon_path and R:
            try: icon_path = R.full_icon_path(icon_path) or icon_path
            except Exception: pass
        logo_td = ""
        if icon_path and os.path.exists(icon_path):
            logo_td = f"<td class='logo'><img src='{icon_path}'/></td>"
        else:
            logo_td = "<td class='logo'></td>"

        # --- Versionenzeile
        if installed:
            meta_line = f"<b>Local:</b> {lv}" + (f" &nbsp; &nbsp; <b>Remote:</b> {rv}" if rv else "")
        else:
            meta_line = f"<b>Remote:</b> {rv}" if rv else ""

        # --- Badges (rechts neben Titel)
        badges = []
        # Non-commercial badge when server-side commercial is not active
        commercial_active = bool(getattr(self.actions, "_commercial_active", False))
        if email and not commercial_active:
            badges.append("<span class='badge badge-no'>Non commercial</span>")

        if not installed:
            badges.append("<span class='badge badge-info'>Not installed</span>")
        else:
            if rv and self._semver_lt(lv, rv):
                badges.append(f"<span class='badge badge-ok'>Update available: {lv} → {rv}</span>")

        if not getattr(self, "_email", None):
            badges.append("<span class='badge badge-no'>Unregistered — free tools only</span>")

        if entitled:
            badges.append("<span class='badge badge-ok'>Entitled</span>")
        else:
            if email and not email_known:
                badges.append("<span class='badge badge-warn'>Not entitled — verify your email</span>")
            else:
                badges.append("<span class='badge badge-warn'>Not entitled</span>")

        if badges:
            cells = "".join(f"<td>{b}</td>" for b in badges)
            badge_html = f"<table class='badges'><tr>{cells}</tr></table>"
        else:
            badge_html = ""


        # --- Header: Logo | (Titel + Version + Badges)
        title = meta.get("title", mid)
        header_html = (
            "<table class='head'><tr>"
            f"{logo_td}"
            "<td>"
            f"<h2 style='margin:0 0 2px 0;'>{title}</h2>"
            f"<div class='meta'>{meta_line}</div>"
            f"{badge_html}"
            "</td>"
            "</tr></table>"
        )

        # --- Optionaler Buy-Link (unterhalb des Headers)
        buy_url  = (man.get("buy_url") or "")
        buy_html = f"<p style='margin:6px 0;'><a href='{buy_url}'>Buy / Learn more</a></p>" if (buy_url and not installed) else ""

        # --- Body
        html = (
            f"{header_html}"
            f"{buy_html}"
            f"<p>{desc}</p>"
            f"<p class='muted'><small>Tags: {tags}</small></p>"
        )
        self.details.setHtml(html)
        self._recalc_buttons()

    def _recalc_buttons(self) -> None:
        it = self.list.currentItem()
        if not it:
            self.btn_primary.setEnabled(False)
            self.btn_primary.setText("Select a tool")
            self._set_variant(self.btn_primary, None)

            self.btn_uninstall.setEnabled(False)
            self._set_destructive(self.btn_uninstall, False)

            self._primary_mode = None
            return

        mid = it.data(QtCore.Qt.UserRole)
        meta = self._catalog_cache.get(mid, {}) or {}

        installed = bool(meta.get("installed"))
        rv = meta.get("remote_version")
        lv = meta.get("version", "0.0.0")
        entitled = (mid in getattr(self, "_allowed_ids", set()))
        has_update = bool(
            installed
            and rv
            and self._semver_lt(lv, rv)
            and (mid in self._updates_index)
        )
        can_install = (not installed) and bool(
            meta.get("manifest_entry", {}).get("zip_url")
        )

        # --- Uninstall-Button Grundzustand ---
        self.btn_uninstall.setEnabled(installed)
        self._set_destructive(self.btn_uninstall, bool(installed))

        # Hub darf NICHT deinstalliert werden
        if mid == "zen_hub":
            self.btn_uninstall.setEnabled(False)
            self._set_destructive(self.btn_uninstall, False)

        # --- Primary Button Logik ---
        if installed:
            if has_update and entitled:
                # UPDATE (gilt auch für zen_hub; Hub darf sich selbst updaten)
                self._primary_mode = "update"
                self.btn_primary.setText("Update selected")
                self.btn_primary.setEnabled(True)
                self._set_variant(self.btn_primary, "primary")

            else:
                # Kein Update verfügbar: normales Tool würde jetzt LAUNCH machen.
                # Für zen_hub überschreiben wir das Verhalten.
                if mid == "zen_hub":
                    # Hub ist offen, kein Launch nötig
                    self._primary_mode = None
                    self.btn_primary.setText("Hub is up to date")
                    self.btn_primary.setEnabled(False)
                    self._set_variant(self.btn_primary, "secondary")

                else:
                    # Standard-"Launch"-Pfad für andere Tools
                    can_launch = callable((get_tools() or {}).get(mid, {}).get("launcher"))
                    self._primary_mode = "launch" if can_launch else None
                    self.btn_primary.setText(
                        "Launch selected" if can_launch else "Please restart Maya"
                    )
                    self.btn_primary.setEnabled(bool(can_launch))
                    self._set_variant(
                        self.btn_primary,
                        "success" if can_launch else None
                    )

        else:
            # NICHT installiert
            if entitled and can_install:
                # INSTALL
                self._primary_mode = "install"
                self.btn_primary.setText("Install selected")
                self.btn_primary.setEnabled(True)
                self._set_variant(self.btn_primary, "primary")

            else:
                # BUY / External link
                self._primary_mode = "buy"
                self.btn_primary.setText("Get this Tool")
                self.btn_primary.setEnabled(True)  # Link öffnet Website
                self._set_variant(self.btn_primary, "secondary")


    def _launch_selected(self) -> None:
        it = self.list.currentItem()
        if not it: return
        mid = it.data(QtCore.Qt.UserRole)
        meta = (get_tools() or {}).get(mid)
        launcher: Optional[Callable] = meta.get("launcher") if meta else None
        if callable(launcher):
            try:
                self._log(f"Launch {mid}"); launcher(); return
            except Exception as e:
                QtWidgets.QMessageBox.critical(self, "Launch failed", str(e))
                self._log(f"Launch error: {e}", level="ERROR")
                print("[ZenToolsHub][Launch] ERROR:\n", traceback.format_exc())
        QtWidgets.QMessageBox.warning(self, "Launch", "No launcher defined for this tool.")
        self._log(f"Launch not possible for {mid}: no launcher", level="WARN")

    def _register_email(self):
        prof = {}
        get_profile = getattr(self.actions, "get_profile", None)
        if callable(get_profile):
            try:
                prof = get_profile() or {}
            except Exception:
                prof = {}
        # Fallbacks
        cur_email = (prof.get("email") or self.actions.email or "").strip().lower()
        cur_name  = (prof.get("name") or "").strip()
        cur_studio = bool(prof.get("studio", False))
        cur_marketing = bool(prof.get("marketing", False))

        ok, payload = LoginDialog.get(parent=self,
                                      name=cur_name,
                                      email=cur_email,
                                      studio=cur_studio,
                                      marketing=cur_marketing)
        if not ok:
            return

        email = payload.get("email", "").strip().lower()
        name = payload.get("name", "").strip()
        studio = bool(payload.get("studio"))
        marketing = bool(payload.get("marketing"))

        ok2, _msg = self.actions.register_email(email, name=name, studio=studio, marketing=marketing)

        def _contains_email(obj, em) -> bool:
            if not em: return False
            if isinstance(obj, str): return obj.strip().lower() == em
            if isinstance(obj, dict): return any(_contains_email(k, em) or _contains_email(v, em) for k, v in obj.items())
            if isinstance(obj, (list, tuple, set)): return any(_contains_email(x, em) for x in obj)
            return False

        if not ok2:
            QtWidgets.QMessageBox.warning(self, "Login", "Login failed. Please try again.")
            return

        ent = self.actions.entitlements or {}
        logged_in = _contains_email(ent, email)
        if logged_in:
            self._show_center_notice("Login", "Successfully logged in.")
        else:
            self._show_center_notice("Register", f"Successfully registered — a confirmation email was sent to {email}.")

        self._reload_catalog()
        self._populate()
        self._refresh_account_button()
        self._rebuild_account_menu()

    def _check_updates(self) -> None:
        cur = self._current_mid()
        found = self.actions.force_check_updates()
        msg = f"{found} update(s) found." if found else "Everything up to date."
        if self.actions is None:
            QtWidgets.QMessageBox.warning(self, "Updates", "Installer-Module not found (core.updater).")
            self._log("Updater missing for check_updates", level="WARN")
            return
        QtWidgets.QMessageBox.information(self, "Updates", msg)
        self._log(f"Check updates: {msg}")
        self._reload_catalog()
        self._populate(prefer_mid=cur)

    def _toggle_updates_popover(self):
        pop = getattr(self, "_updates_popover", None)
        if pop and pop.isVisible():
            pop.close()
            return
        self._show_updates_popover()

    def _show_updates_popover(self):
        # Build popover shell
        self._updates_popover = ArrowPopover(self)
        self._updates_popover.closed.connect(lambda: setattr(self, "_updates_popover", None))

        # Content
        w = QtWidgets.QWidget()
        v = QtWidgets.QVBoxLayout(w)
        v.setContentsMargins(0, 0, 0, 0)
        v.setSpacing(8)

        title = QtWidgets.QLabel("Check for updates")
        title.setStyleSheet("font-size:14px; font-weight:600;")

        # Status line (changes from "Checking…" to the result)
        self._updates_status_lbl = QtWidgets.QLabel("Checking…")
        self._updates_status_lbl.setProperty("role", "muted")

        # Hint line (always visible; text will be updated after the check)
        self._updates_hint_lbl = QtWidgets.QLabel("This may take a moment.")
        self._updates_hint_lbl.setProperty("role", "muted")

        v.addWidget(title)
        v.addWidget(self._updates_status_lbl)
        v.addWidget(self._updates_hint_lbl)

        self._updates_popover.setContentWidget(w)
        self._updates_popover.resize(280, 120)
        self._updates_popover.popupAt(self.btn_check_icon, dx=-20, gap=6, clamp_to_parent=False)

        # Kick off background check in a thread
        self._updates_thread = QtCore.QThread(self)
        self._updates_worker = _CheckUpdatesWorker(self.actions)
        self._updates_worker.moveToThread(self._updates_thread)

        self._updates_thread.started.connect(self._updates_worker.run)
        self._updates_worker.finished.connect(self._on_updates_finished)
        self._updates_worker.finished.connect(self._updates_thread.quit)
        self._updates_worker.finished.connect(self._updates_worker.deleteLater)
        self._updates_thread.finished.connect(self._updates_thread.deleteLater)
        self._updates_thread.start()

    def _on_updates_finished(self, count: int, error: str):
        # Popover may be closed already
        if not getattr(self, "_updates_popover", None):
            return

        if error:
            self._updates_status_lbl.setText(f"Error: {error}")
            if getattr(self, "_updates_hint_lbl", None):
                self._updates_hint_lbl.setText("Please try again.")
        else:
            if count <= 0:
                self._updates_status_lbl.setText("Everything up to date.")
                # Keep hint visible, but make it meaningful post-check:
                if getattr(self, "_updates_hint_lbl", None):
                    self._updates_hint_lbl.setText("You're on the latest version.")
            else:
                plural = "update" if count == 1 else "updates"
                self._updates_status_lbl.setText(f"{count} {plural} available.")
                # Mirror the count in the hint line as requested:
                if getattr(self, "_updates_hint_lbl", None):
                    self._updates_hint_lbl.setText(f"Found {count} {plural}. Use the list to update.")

        # Refresh list to reflect any updated catalog/index
        cur = self._current_mid()
        self._reload_catalog()
        self._populate(prefer_mid=cur)



    def _auto_check_updates_on_open(self, initial: bool = False) -> None:
        cur = self._current_mid()

        # 1) Lokale Installation scannen (welche Tools liegen am Disk)
        #    => setzt installed, pfade usw. in actions.catalog_cache (lokal)
        try:
            if initial:
                self.actions.rescan()
        except Exception as e:
            self._log(f"Rescan failed: {e}", level="WARN")

        # 2) Remote / Entitlements / Updates mergen
        #    => füllt catalog_cache endgültig (inkl. installed=True für vorhandene)
        try:
            found = self.actions.force_check_updates()
            self._log(f"Auto check updates: {found} possible update(s).")
        except Exception as e:
            self._log(f"Auto check updates failed: {e}", level="WARN")

        # 3) UI-Daten rüberziehen + Liste füllen
        self._reload_catalog()
        self._populate(prefer_mid=cur)



    def _update_selected(self) -> None:
        it = self.list.currentItem()
        if not it:
            return
        cur = self._current_mid()
        mid = it.data(QtCore.Qt.UserRole)

        self.btn_primary.setEnabled(False)
        self.btn_primary.setText("Updating…")
        QtWidgets.QApplication.processEvents()

        ok, msg = self.actions.update_selected(mid)
        if ok:
            QtWidgets.QMessageBox.information(
                self,
                "Update",
                "Update completed.\n\nPlease restart Maya to apply changes."
            )
        else:
            QtWidgets.QMessageBox.warning(self, "Update failed", msg or "Update failed.")

        self.btn_primary.setText("Update selected")
        self.btn_primary.setEnabled(True)
        self._reload_catalog()
        self._populate(prefer_mid=cur)

    def _get_profile(self) -> dict:
        prof = {}
        get_profile = getattr(self.actions, "get_profile", None)
        if callable(get_profile):
            try: prof = get_profile() or {}
            except Exception: prof = {}
        if not prof:
            em = (getattr(self.actions, "email", "") or "").strip().lower()
            if em:
                prof = {"email": em, "name": "", "studio": False, "marketing": False}
        return prof or {}

    def _profile_exists(self) -> bool:
        p = self._get_profile()
        em = (p.get("email") or "").strip()
        return bool(em and "@" in em)

    def _refresh_account_button(self):
        prof_exists = self._profile_exists()
        try:
            self.btn_account.setToolTip("My Account" if prof_exists else "Login / Register")
            # Optional: swap the glyph if you want (kept as 👤 for both states)
        except Exception:
            pass

    def _login_or_account(self):
        if not self._profile_exists():
            # Kein Profil → regulärer Login-Flow (deine bestehende Methode)
            self._register_email()
            # Danach Button-Text anpassen
            self._refresh_account_button()
            return

        # Profil existiert → Read-only Dialog anzeigen
        prof = self._get_profile()
        dlg = AccountDialog(parent=self, profile=prof)
        is_logout = dlg.exec_() == QtWidgets.QDialog.Accepted
        if not is_logout:
            return

        # Logout ausführen
        self._perform_logout()
        self._refresh_account_button()
        # Optional UI-Refresh
        self._reload_catalog()
        self._populate()

    def _perform_logout(self):
        # 1) Bevorzugt: Actions.logout()
        for fn_name in ("logout", "clear_profile", "sign_out"):
            fn = getattr(self.actions, fn_name, None)
            if callable(fn):
                try:
                    ok = fn()
                    if ok is None or ok is True:
                        return
                except Exception:
                    pass

        # 2) Fallback: generische Utility-Funktionen, falls vorhanden
        for fn_name in ("_delete_license_file", "_delete_entitlement_cache"):
            fn = getattr(self.actions, fn_name, None)
            if callable(fn):
                try: fn()
                except Exception: pass

        # 3) Letzter Fallback: Neu-Registrierung mit leerer E-Mail (deaktiviert Account lokal)
        try:
            reg = getattr(self.actions, "register_email", None)
            if callable(reg):
                reg("")  # ignoriert der Server; lokal sollte _save_license fehlschlagen/neutralisieren
        except Exception:
            pass

    def _install_selected(self) -> None:
        it = self.list.currentItem()
        if not it:
            return
        cur = self._current_mid()
        mid = it.data(QtCore.Qt.UserRole)
        meta = self._catalog_cache.get(mid, {})
        if meta.get("installed"):
            return

        self.btn_primary.setEnabled(False)
        self.btn_primary.setText("Installing…")
        QtWidgets.QApplication.processEvents()
        ok, msg = self.actions.install_selected(mid)
        self._reload_catalog()
        new_meta = self._catalog_cache.get(mid, {}) or {}
        path = new_meta.get("path") or new_meta.get("install_path") or new_meta.get("install_dir") or ""
        if not path:
            try:
                import maya.cmds as _cmds  # type: ignore
                scripts_dir = _cmds.internalVar(userScriptDir=True).replace("\\", "/")
                path = os.path.join(scripts_dir, f"pjots_zen_tools/tools/{mid}").replace("\\", "/")
            except Exception:
                path = f"<pjots_zen_tools/tools/{mid}>"

        if ok:
            text = f"Installation completed.\n\nInstalled to:\n{path}"
            QtWidgets.QMessageBox.information(self, "Install", text)
        else:
            QtWidgets.QMessageBox.warning(self, "Install failed", msg or "Installation failed.")

        self.btn_primary.setText("Install selected")
        self.btn_primary.setEnabled(True)
        self._populate(prefer_mid=cur)


    def _uninstall_selected(self) -> None:
        it = self.list.currentItem()
        if not it: return
        cur = self._current_mid()
        mid = it.data(QtCore.Qt.UserRole)
        meta = self._catalog_cache.get(mid, {})
        if not meta.get("installed"): return
        confirm = QtWidgets.QMessageBox.question(
            self, "Uninstall",
            f"Shall '{meta.get('title', mid)}' be uninstalled?",
            QtWidgets.QMessageBox.Yes | QtWidgets.QMessageBox.No, QtWidgets.QMessageBox.No)
        if confirm != QtWidgets.QMessageBox.Yes:
            self._log("Uninstall canceled by user"); self._recalc_buttons(); return
        ok, msg = self.actions.safe_uninstall(mid)
        QtWidgets.QMessageBox.information(self, "Uninstall" if ok else "Uninstall failed", msg)
        self._reload_catalog()
        # Falls das Tool weg ist, bleibt prefer_mid wirkungslos → Row 0
        self._populate(prefer_mid=cur)

class AccountDialog(QtWidgets.QDialog):
    def __init__(self, parent=None, profile=None):
        super().__init__(parent)
        self.setObjectName("ZenRoot")
        apply_styles(self)
        self.setWindowTitle("My Account")
        self.setModal(True)
        self.setMinimumWidth(400)
        prof = profile or {}
        v = QtWidgets.QVBoxLayout(self)
        v.setContentsMargins(12, 12, 12, 12)
        v.setSpacing(8)
        form = QtWidgets.QFormLayout()
        form.setLabelAlignment(QtCore.Qt.AlignRight)
        le_name = QtWidgets.QLineEdit(prof.get("name", ""))
        le_email = QtWidgets.QLineEdit(prof.get("email", ""))
        le_studio = QtWidgets.QLineEdit("Yes" if prof.get("studio") else "No")
        le_marketing = QtWidgets.QLineEdit("Yes" if prof.get("marketing") else "No")
        for w in (le_name, le_email, le_studio, le_marketing):
            w.setReadOnly(True)

        form.addRow("Name:", le_name)
        form.addRow("Email:", le_email)
        form.addRow("Commercial license:", le_studio)
        form.addRow("Marketing opt-in:", le_marketing)
        v.addLayout(form)
        v.addSpacing(8)
        v.addWidget(QtWidgets.QLabel("To switch account or remove your license from this machine, click Logout."))
        btns = QtWidgets.QDialogButtonBox()
        self.btn_logout = btns.addButton("Logout", QtWidgets.QDialogButtonBox.AcceptRole)
        self.btn_cancel = btns.addButton("Cancel", QtWidgets.QDialogButtonBox.RejectRole)
        v.addWidget(btns)
        btns.accepted.connect(self.accept)
        btns.rejected.connect(self.reject)

# ---------------------------------------------------------------------
# Public API
# ---------------------------------------------------------------------
_HUB = None
def show():
    global _HUB
    if _HUB and _HUB.isVisible():
        _HUB.raise_(); _HUB.activateWindow(); return _HUB
    _HUB = ZenToolsHub(parent=None)
    _HUB.show()
    return _HUB
