# -*- coding: utf-8 -*-
"""
Discovery & registration for Pjots Zen Tools.

Enhancements:
- Registers the Hub itself (module_id "zen_hub") in addition to tools/* packages
- Accepts hub_meta either as dict META or as module-level fields (title, version, ...)
- Supports launcher as callable or "pkg.mod:func" string (also "entry")
- Normalizes icon paths relative to the providing module's folder
"""
from __future__ import annotations

import os
import importlib
import pkgutil
from typing import Optional, Dict, Any, Callable

from .registry import register_tool

PKG_NAME = "pjots_zen_tools"
TOOLS_PKG = f"{PKG_NAME}.tools"

# ----------------------------- helpers -----------------------------

def _safe_import(module_name: str):
    try:
        return importlib.import_module(module_name)
    except Exception:
        return None


def _build_launcher(launcher_ref: Any) -> Optional[Callable]:
    """Return a callable from a reference or a "pkg.mod:func" string."""
    if callable(launcher_ref):
        return launcher_ref
    if isinstance(launcher_ref, str):
        mod, _, func = launcher_ref.partition(":")
        if mod and func:
            m = _safe_import(mod)
            if m and hasattr(m, func):
                fn = getattr(m, func)
                if callable(fn):
                    return fn
    return None


def _norm_icon_path(mod, icon_value: str) -> str:
    """Return absolute icon path if icon_value is relative and module has __file__."""
    if not icon_value:
        return ""
    if os.path.isabs(icon_value):
        return icon_value
    base = os.path.dirname(getattr(mod, "__file__", ""))
    if base:
        p = os.path.join(base, icon_value)
        return p
    return icon_value


def _as_meta_dict(mod) -> Optional[Dict[str, Any]]:
    """Extract a normalized META dict from a module which may provide either
    a dict `META` or separate variables (title, version, description, tags, icon, launcher, module_id).
    """
    # Preferred: dict META
    meta = getattr(mod, "META", None)
    if isinstance(meta, dict):
        m = dict(meta)
        m.setdefault("module_id", m.get("id"))
        if "icon" in m:
            m["icon"] = _norm_icon_path(mod, m.get("icon", ""))
        # Support alternate key 'entry' for launcher string
        if not m.get("launcher") and isinstance(m.get("entry"), str):
            m["launcher"] = m["entry"]
        return m

    # Fallback: module-level fields
    title = getattr(mod, "title", None)
    version = getattr(mod, "version", None)
    description = getattr(mod, "description", None)
    tags = getattr(mod, "tags", None)
    icon = getattr(mod, "icon", None)
    launcher = getattr(mod, "launcher", None)
    module_id = getattr(mod, "module_id", None)

    if any(v is not None for v in (title, version, description, tags, icon, launcher, module_id)):
        m = {
            "module_id": module_id,
            "title": title,
            "version": version or "0.0.0",
            "description": description or "",
            "tags": list(tags or []),
            "icon": _norm_icon_path(mod, icon or ""),
            "launcher": launcher,
        }
        return m

    return None


def _register_from_meta(mod) -> bool:
    """Register a tool from a hub_meta-like module.
    Supports dict META or discrete module-level fields.
    """
    meta = _as_meta_dict(mod)
    if not isinstance(meta, dict):
        return False

    module_id = meta.get("module_id") or meta.get("id")
    if not module_id:
        return False

    launcher = _build_launcher(meta.get("launcher"))
    if not launcher:
        return False

    register_tool(
        module_id=module_id,
        title=meta.get("title", module_id),
        launcher=launcher,
        icon=meta.get("icon", ""),
        version=meta.get("version", "0.0.0"),
        tags=meta.get("tags", []),
        description=meta.get("description", ""),
    )
    return True


# ----------------------------- public API -----------------------------

def register_hub_self(module_id: str = "zen_hub") -> None:
    """Ensure the Hub itself is registered in the local registry under `module_id`.
    Priority: hub.hub_meta → hub (META) → hardcoded defaults.
    """
    # 1) hub.hub_meta
    mod = _safe_import(f"{PKG_NAME}.hub.hub_meta")
    if mod and _register_from_meta(mod):
        return

    # 2) hub package exposing META
    hub_pkg = _safe_import(f"{PKG_NAME}.hub")
    if hub_pkg:
        meta = getattr(hub_pkg, "META", None)
        if isinstance(meta, dict):
            # ensure module_id present
            meta = dict(meta)
            meta.setdefault("module_id", module_id)
            launcher = _build_launcher(meta.get("launcher") or f"{PKG_NAME}.hub:show")
            if launcher:
                register_tool(
                    module_id=meta.get("module_id", module_id),
                    title=meta.get("title", "Zen Tools Hub"),
                    launcher=launcher,
                    icon=_norm_icon_path(hub_pkg, meta.get("icon", "")),
                    version=meta.get("version", "0.0.0"),
                    tags=meta.get("tags", []),
                    description=meta.get("description", ""),
                )
                return

    # 3) Hardcoded safe default
    launcher = _build_launcher(f"{PKG_NAME}.hub:show")
    if launcher:
        # try resolving a default icon next to hub package
        icon = _norm_icon_path(hub_pkg or __import__(PKG_NAME + ".hub", fromlist=["*"]), "icons/pjots_zen_tools_hub.png")
        register_tool(
            module_id=module_id,
            title="Zen Tools Hub",
            launcher=launcher,
            icon=icon,
            version="0.0.0",
            tags=["hub"],
            description="Zentrale Verwaltungsoberfläche.",
        )


def discover_and_register():
    """Discover tools and register them into the local registry.
    Includes:
    - Self-register the Hub
    - Discover packages under pjots_zen_tools.tools.*
    - Prefer hub_meta per tool; fallback to package import side-effects
    """
    # Make sure the Hub shows up in the list
    try:
        register_hub_self()
    except Exception:
        pass

    # Discover tools/* packages
    pkg = _safe_import(TOOLS_PKG)
    if not pkg or not hasattr(pkg, "__path__"):
        return

    for _, name, ispkg in pkgutil.iter_modules(pkg.__path__, prefix=f"{TOOLS_PKG}."):
        if not ispkg:
            continue
        # 1) hub_meta preferred (no __init__ side-effects)
        meta_mod = _safe_import(f"{name}.hub_meta")
        if meta_mod and _register_from_meta(meta_mod):
            continue
        # 2) Fallback: import the package (it may self-register)
        _safe_import(name)
